/**
Copyright (C) 2011-2013 Stefan Kolb.
Copyright (C) 2013 Moritz Stoll (shearing box boundary conditions)
Copyright (C) 2019 Salvatore Colombo (non-LTE equations)
 
This file is part of the radiation module for the code PLUTO.

The radiation module is free software: you can redistribute it
and/or modify it under the terms of the GNU General Public
License as published by the Free Software Foundation, either
version 2 of the License, or (at your option) any later version.

The radiation module is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty
of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with the radiation module. If not, see <http://www.gnu.org/licenses/>.
*/

#include <string.h>
#include <execinfo.h>
#include <signal.h>
#include <stdlib.h>
#include <limits.h>
#define __USE_ISOC99
#include <math.h>

#ifndef __USE_ISOC99
    /*
        This defined enables some macros in math.h it is needed only on some machines.
        We need it here for the macro isnormal(x)
    */
    #define __USE_ISOC99
#endif
#include <math.h>

#include "radiation.h"
#include "modify_pluto.h"
#include "radiation_tools.h"
#include "opacity.h"
#include "assert.h"

#if RADIATION_SOLVER == PETSC_LIBRARY
    #include "petsc.h"
#endif

#if IRRADIATION == YES
    #include "irradiation.h"
#endif

#include "SemenovOpacity/semenov_opacity.h"

#if DIMENSIONS != 3
    #error Dimension must be set to 3 if you want to use the radiation module!
#endif

#ifndef PARALLEL
    #error The radiation module must be compuled with a makefile where PARALLEL=TRUE is set like Linux.mpicc.defs
#endif

RadiationData radiation_data = {.code_c = 1.0, .code_mH = 1.0, .code_amu = 1.0, .code_aR = 1.0, .code_kB = 1.0, .mu = 0.6, .Erad = NULL, .default_init_Erad = 1};

double g_unitTemperature = 1.;  /**<Transformation between temperature in code units and temperature in kelvin \note is set in \ref RadiationInit()*/

/**
    With this function it is possible to change the initial behavior of Erad. If \a b is 1 then \ref RadiationSetInitialErad is called in function
    \ref RadiationInit else not. This function is only needed if Erad is set by the user or if Erad is loaded from a file.

    \param[in] b
*/
void RadiationSetEradInitBehaviour( int b )
{
    radiation_data.default_init_Erad = b;
}

/**
    Sets \f$ \mu \f$ for the computation

    \param[in] mu
*/
void RadiationSetMu( double mu )
{
    radiation_data.mu = mu;
}

/**
    Prints out the current set boundary conditions
*/
void PrintBoundaryConditions()
{
    int i = 0;
    char *boundary_string[] = {"invalid", "periodic", "symmetric", "reflective", "outflow", "fixedvalue", "shearingbox"};

    if( prank == 0 )
    {
        print( "> Radiation boundary conditions\n" );
        for( i = 0; i < 3; ++i )
        {
            print( " X%i boundary: [beg ... end]  %10s ... %10s\n", i + 1, boundary_string[radiation_data.boundary_conditions[i].bc_beg], boundary_string[radiation_data.boundary_conditions[i].bc_end] );
            if( radiation_data.boundary_conditions[i].bc_beg == RADIATION_BC_FIXEDVALUE )
            {
                print( "    begin fixedvalue=%10.3e fixedvalueT=%10.3f\n", radiation_data.boundary_conditions[i].beg_fixedvalue * ( g_unitDensity * POW2( g_unitVelocity ) ), pow( radiation_data.boundary_conditions[i].beg_fixedvalue / radiation_data.code_aR, 1. / 4. )*g_unitTemperature );
            }

            if( radiation_data.boundary_conditions[i].bc_end == RADIATION_BC_FIXEDVALUE )
            {
                print( "    end fixedvalue=%10.3e fixedvalueT=%10.3f\n", radiation_data.boundary_conditions[i].end_fixedvalue * ( g_unitDensity * POW2( g_unitVelocity ) ), pow( radiation_data.boundary_conditions[i].end_fixedvalue / radiation_data.code_aR, 1. / 4. )*g_unitTemperature );
            }
        }
    }
}

/**
    Print some units and constants which can be used for conversion the code units back to
    cgs-units in post processing tools.

    \note Some of the units are printed out by PLUTO but we print it again with higher precision
*/
void RadiationPrintUnits()
{
    if( prank == 0 )
    {
        print( "\n> Normalisation Units for postprocessing\n" );
        print( " %-25s %-15.10e %s\n", "[g_unitDensity]:", g_unitDensity, "(gr/cm^3)" );
        print( " %-25s %-15.10e %s\n", "[g_unitVelocity]:", g_unitVelocity, "(cm/s)" );
        print( " %-25s %-15.10e %s\n", "[g_unitLength]:", g_unitLength, "(cm)" );
        print( " %-25s %-15.10e %s\n", "[g_unitTemperature]:", g_unitTemperature, "(K)" );

        print( "\n> Constants needed for postprocessing\n" );
        print( " %-25s %-15.10e %s\n", "[mu]:", radiation_data.mu, "" );
        print( " %-25s %-15.10e %s\n", "[gamma]:", g_gamma, "" );
        print( "\n" );

//      print(" %-25s %-15.10e %s\n","[CONST_amu]:",CONST_amu,"(gr)");

//      print("code_c=%15.10e CONST_c=%15.10e\n",radiation_data.code_c,CONST_c);
//      print("code_aR=%15.10e CONST_aR=%15.10e\n",radiation_data.code_aR,CONST_aR);
//      print("code_kB=%15.10e CONST_kB=%15.10e\n",radiation_data.code_kB,CONST_kB);
//      print("code_mH=%15.10e CONST_mH=%15.10e\n",radiation_data.code_mH,CONST_mH);
    }
}

/**
    Function to handle segmentation faults. This means if setup the function is called
    if a segmentation fault occurs.
*/
void SegfaultHandleFunction( int sig )
{
    const int buffer_size = 30;
    void *buffer[buffer_size];
    int size = 0;

    size = backtrace( buffer, buffer_size );
    fprintf( stderr, "Error: signal %d:\n", sig );
    backtrace_symbols_fd( buffer, size, 2 ); //prints the functions to file descriptor 2 (stderr)
    exit( 1 );
}

/**
    Calculates the new temperature \f$ T_{i,j,k}^{n+1} \f$ by using the radiation energy

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  grid
    \param[inradiation_data.code_c]  data

    \returns
        In the case that irradiation and/or artificial dissipation is used this function returns
        \f[
            T_{i,j,k}^{n+1} = \frac{{\kappa_P}_{i,j,k}^n c \left(3 a_R (T_{i,j,k}^n)^4  + E_{i,j,k}^{n+1}\right)\Delta t + \frac{W^n_{i,j,k} \Delta t}{\rho^n_{i,j,k}} + c_V T_{i,j,k}^n}{c_V + 4 {\kappa_P}_{i,j,k}^n c a_R (T_{i,j,k}^n)^3 \Delta t}
        \f]
        where \f$ W^n_{i,j,k} \f$ is either \f$ S^n_{i,j,k} \f$ in the case of irradiation or \f$ D^n_{i,j,k} \f$ in the case of artificial dissipation. If irradiation and artificial dissipation is used together then \f$ W^n_{i,j,k} \f$ is the summ of both \f$ W^n_{i,j,k} = S^n_{i,j,k}+D^n_{i,j,k}\f$.\n
        else
        \f[
            T_{i,j,k}^{n+1} = \frac{{\kappa_P}_{i,j,k}^n c \left(3 a_R (T_{i,j,k}^n)^4  + E_{i,j,k}^{n+1}\right)\Delta t + c_V T_{i,j,k}^n}{c_V + 4 {\kappa_P}_{i,j,k}^n c a_R (T_{i,j,k}^n)^3 \Delta t}
        \f]
        \f$ E_{i,j,k}^{n+1} \f$ labels the radiation energy at the time step n+1

*/
double ComputeNewTemperature( int k, int j, int i, Grid *grid, Data *data )
{

    #if IRRADIATION == YES || ARTIFICIAL_DISSIPATION == YES
        double plank_opacity = ComputePlanckOpacity( k, j, i, data, grid );
        double temperature = GetTemperature( k, j, i, data );
        double c_V = ComputeSpecificHeatCapacity( k, j, i, data );
        double rho = data->Vc[RHO][k][j][i];

        double W = 0.0;
        #if IRRADIATION == YES
            W = irradiation.S[k][j][i];
        #endif

        #if ARTIFICIAL_DISSIPATION == YES
            W += ComputeArtificialDissipation( k, j, i, data, grid );
        #endif

        return ( plank_opacity * radiation_data.code_c * ( 3. * radiation_data.code_aR * POW4( temperature ) + radiation_data.Erad[k][j][i] ) * g_dt + ( W * g_dt ) / rho + c_V * temperature ) / ( c_V + 4. * plank_opacity * radiation_data.code_c * radiation_data.code_aR * POW3( temperature ) * g_dt );
    #else
        double rho = data->Vc[RHO][k][j][i];
        double plank_opacity = ComputePlanckOpacity( k, j, i, data, grid );
        double temperature = GetTemperature( k, j, i, data );
        double c_V = ComputeSpecificHeatCapacity( k, j, i, data );
        double delta_t = g_dt/(g_unitVelocity/g_unitLength);
        double pwl = ComputePowerlosses( k, j, i, data, grid );
	double d_pwl = ComputeDerivativePowerlosses( k, j, i, data, grid );

        double Ener = radiation_data.Erad[k][j][i];        
        double coef1 = delta_t/(rho*c_V);
        double coef2 = plank_opacity*rho*radiation_data.code_c*Ener - pwl;
        double denom = 1. + coef1*d_pwl;
    
        double TempK;
        // NLTE
    
		Temp_res = temperature + coef1/(denom) *coef2;
    	return Temp_res;
    #endif
}

/**
    Updates the pressure in PLUTO after the implicit radiation step

    The temperature is given by \f$ T = \frac{p}{\rho} \frac{\mu m_u}{k_B}\f$ where \f$ \mu m_u \f$ the mass
    of one gas molecule, so we get for the pressure \f$ p = T \rho \frac{k_B}{\mu m_u} \f$.


    \note It is not necessary to call function \a AL_Exchange_dim because it was called for the array Erad before, so the ghost cell for Erad are set correctly

    \param[in]  grid
    \param[in]  data
    \param[out] data
*/
void UpdatePlutoPressure( Grid *grid, Data *data )
{
    int k = 0, j = 0, i = 0;
    double *** pr = data->Vc[PRS];
    double *** rho = data->Vc[RHO];
    double temperature = 0.;

    TOT_LOOP( k, j, i )
    {
        temperature = ComputeNewTemperature( k, j, i, grid, data );
        pr[k][j][i] = ( temperature * rho[k][j][i] * radiation_data.code_kB ) / ( radiation_data.mu * radiation_data.code_amu );
    }
}

/**
    Sets the initial values for the radiation energy to \f$ E_r = a_R T^4 \f$ the energy of a black body at the temperature \a T.

    \param[in]  grid
    \param[in]  data
    \param[out] data
*/
void RadiationSetInitialEradBlackBody( Grid *grid, Data *data )
{
    int k, j, i;

    double temperature = 0.;

    TOT_LOOP( k, j, i )
    {
        temperature = GetTemperature( k, j, i, data );
        radiation_data.Erad[k][j][i] = radiation_data.code_aR * POW4( temperature );
    }
}

/**
    Sets the fixed value boundary conditions

    \param[in]  q   is a pointer to the array containing the variable. For example data->Vc[RHO]
    \param[in]  side
    \param[out] q
*/
void FixedValueBoundaryCondition( double ***q, RBox *box, int side )
{
    int k, j, i;
    switch( side )
    {    
        case X1_BEG:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = radiation_data.boundary_conditions[IDIR].beg_fixedvalue;
            }
            break;

        case X1_END:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = radiation_data.boundary_conditions[IDIR].end_fixedvalue;
            }
            break;

        case X2_BEG:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = radiation_data.boundary_conditions[JDIR].beg_fixedvalue;
            }
            break;

        case X2_END:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = radiation_data.boundary_conditions[JDIR].end_fixedvalue;
            }
            break;

        case X3_BEG:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = radiation_data.boundary_conditions[KDIR].beg_fixedvalue;
            }
            break;

        case X3_END:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = radiation_data.boundary_conditions[KDIR].end_fixedvalue;
            }
            break;
    }
}

/**
    Sets the reflective value boundary conditions

    \param[in]  q   is a pointer to the array containing the variable. For example data->Vc[RHO]
    \param[in]  side
    \param[out] q
*/
void ReflectiveBoundaryCondition( double ***q, RBox *box, int side )
{
    int k, j, i;
    switch( side )
    {
        case X1_BEG:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = q[k][j][2 * IBEG - i - 1];
            }
            break;

        case X1_END:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = q[k][j][2 * IEND - i + 1];
            }
            break;

        case X2_BEG:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = q[k][2 * JBEG - j - 1][i];
            }
            break;

        case X2_END:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = q[k][2 * JEND - j + 1][i];
            }
            break;

        case X3_BEG:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = q[2 * KBEG - k - 1][j][i];
            }
            break;

        case X3_END:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = q[2 * KEND - k + 1][j][i];
            }
            break;
    }
}

/**
    Sets the periodic value boundary conditions

    \param[in]  q   is a pointer to the array containing the variable. For example data->Vc[RHO]
    \param[in]  side
    \param[out] q
*/
void PeriodicBoundaryCondition( double ***q, RBox *box, int side )
{
    int k, j, i;
    switch( side )
    {
        case X1_BEG:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = q[k][j][i + NX1];
            }
            break;

        case X1_END:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = q[k][j][i - NX1];
            }
            break;

        case X2_BEG:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = q[k][j + NX2][i];
            }
            break;

        case X2_END:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = q[k][j - NX2][i];
            }
            break;

        case X3_BEG:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = q[k + NX3][j][i];
            }
            break;

        case X3_END:
            BOX_LOOP( box, k, j, i )
            {
                q[k][j][i] = q[k - NX3][j][i];
            }
            break;
    }
}

/**
    Sets the boundary conditions for the radiation energy.

    Exchange boundaries between neighbor processes, for this purpose the function \ref AL_Exchange_dim is used.

    \note For more information look at boundary.c and $PLUTO_DIR/Lib/ArrayLib/src/al_exchange_dim.c

    \param[in]  grid
    \param[in]  data
    \param[out] data
*/
void RadiationBoundaryCondition( Grid *grid, Data *data )
{
    int parallel_dimension[3] = {0, 0, 0};
    int physical_boundary[6];
    int side[6] = {X1_BEG, X1_END, X2_BEG, X2_END, X3_BEG, X3_END};
    int is = 0;
    static int do_once = 1;
    static RBox center[8], x1face[8], x2face[8], x3face[8];

    if( do_once )
    {
        SetRBox( center, x1face, x2face, x3face );
        do_once = 0;
    }

    //check how many processes are available in each direction
    parallel_dimension[0] = grid[IDIR].nproc > 1;
    parallel_dimension[1] = grid[JDIR].nproc > 1;
    parallel_dimension[2] = grid[KDIR].nproc > 1;

    //exchange data between processes
    AL_Exchange_dim( ( char * )radiation_data.Erad[0][0], parallel_dimension, SZ ); //SZ seems to be a descriptor for ArrayLib

    physical_boundary[0] = grid[IDIR].lbound > 0 ? radiation_data.boundary_conditions[IDIR].bc_beg : RADIATION_BC_NO_BOUNDARY;
    physical_boundary[1] = grid[IDIR].rbound > 0 ? radiation_data.boundary_conditions[IDIR].bc_end : RADIATION_BC_NO_BOUNDARY;

    physical_boundary[2] = grid[JDIR].lbound > 0 ? radiation_data.boundary_conditions[JDIR].bc_beg : RADIATION_BC_NO_BOUNDARY;
    physical_boundary[3] = grid[JDIR].rbound > 0 ? radiation_data.boundary_conditions[JDIR].bc_end : RADIATION_BC_NO_BOUNDARY;

    physical_boundary[4] = grid[KDIR].lbound > 0 ? radiation_data.boundary_conditions[KDIR].bc_beg : RADIATION_BC_NO_BOUNDARY;
    physical_boundary[5] = grid[KDIR].rbound > 0 ? radiation_data.boundary_conditions[KDIR].bc_end : RADIATION_BC_NO_BOUNDARY;

    for( is = 0; is < 6; ++is )
    {
        switch( physical_boundary[is] )
        {
            case RADIATION_BC_NO_BOUNDARY:
                continue;
                break;

            case RADIATION_BC_PERIODIC:
                /*
                    PERIODIC_BOUND is only called if there's one processor in this direction.

                    If there is more than one processor in this direction the periodic boundaries
                    are set by the Array library (by the function AL_Exchange_dim called above)
                */
                if( parallel_dimension[is / 2] == 0 )
                {
                    PeriodicBoundaryCondition( radiation_data.Erad, center + is, side[is] );
                }

                break;

            case RADIATION_BC_SYMMETRIC:
            case RADIATION_BC_REFLECTIVE:
            case RADIATION_BC_OUTFLOW:
                ReflectiveBoundaryCondition( radiation_data.Erad, center + is, side[is] );
                break;

            case RADIATION_BC_FIXEDVALUE:
                FixedValueBoundaryCondition( radiation_data.Erad, center + is, side[is] );
                break;
            #ifdef SHEARINGBOX
                case RADIATION_BC_SHEARINGBOX:
                {
                    RBox box;
                    if( side[is] == X1_BEG )
                    {
                        box.ib = 0; box.ie = IBEG - 1;
                        box.jb = 0; box.je = NX2_TOT - 1;
                        box.kb = 0; box.ke = NX3_TOT - 1;
                    }

                    if( side[is] == X1_END )
                    {
                        box.ib = IEND + 1; box.ie = NX1_TOT - 1;
                        box.jb = 0; box.je = NX2_TOT - 1;
                        box.kb = 0; box.ke = NX3_TOT - 1;
                    }

                    if( parallel_dimension[is / 2] == 0 )
                    {
                        PeriodicBoundaryCondition( radiation_data.Erad, center + is, side[is] );
                    }

                    SB_SetBoundaryVar( radiation_data.Erad, &box, side[is] , g_time , grid );
                }
                break;
            #endif

        }
    }
}

/**
    In this function we read the informations for the boundary conditions out of the
    file pluto.ini and create a global array which provides this information per direction.

    A example section for the boundary conditions in the file pluto.ini look like:
    \verbatim
    [Radiation-Boundary]
    rX1-beg        fixedvalue 1e-8
    rX1-end        fixedvalueT 50.7
    rX2-beg        reflective
    rX2-end        symmetric
    rX3-beg        periodic
    rX3-end        periodic
    \endverbatim

    We implemented four different boundary conditions for now periodic,symmetric,reflective
    and fixedvalue. If you specify the boundary condition fixedvalue you have to set the radiation energy
    like "rX1-beg fixedvalue 1e-8" if the value is 1e-8. There is also the keyword fixedvalueE which is the
    same as fixedvalue. If you want to set the radiation energy corresponding to a temperature you can use
    fixedvalueT in the same way as fixedvalue. In this case the temperature is converged to the radiation
    energy by the formula \f$ E = a_R T^4 \f$ (\f$ E \f$ = the radiation energy).

    \note
        For a scalar reflective and symmetric are the same boundary condition (this is the case here)   \n
        For informations about \ref ParGet see \ref parse_file.c and setup.c

    \param[in] grid
*/
void SetupBoundaryConditions( Grid *grid )
{
    char option[256];
    char *result;
    int pos = 0, dir = 0, begin = 0;

    if( prank == 0 )
    {
        for( pos = 0; pos < 6; pos++ )
        {
            if( pos < 3 )
            {
                sprintf( option, "rX%d-beg", pos + 1 );
                dir = pos;
                begin = 1;
                radiation_data.boundary_conditions[dir].beg_fixedvalue = 0.; //set it to a defined value for the case that it is not used
            }
            else
            {
                sprintf( option, "rX%d-end", pos - 2 );
                dir = pos - 3;
                begin = 0;
                radiation_data.boundary_conditions[dir].end_fixedvalue = 0.; //set it to a defined value for the case that it is not used
            }

            result = ParGet( option, 1 );
            if( !result )
            {
                printf( "Option %s not found in pluto.ini\n", option );
                QUIT_PLUTO( 1 );
            }

//          printf("%s %s\n",option,result);

            if( strcmp( result, "periodic" ) == 0 )
            {
                if( begin )
                {
                    radiation_data.boundary_conditions[dir].bc_beg = RADIATION_BC_PERIODIC;
                }
                else
                {
                    radiation_data.boundary_conditions[dir].bc_end = RADIATION_BC_PERIODIC;
                }
            }
            else if( strcmp( result, "symmetric" ) == 0 )
            {
                if( begin )
                {
                    radiation_data.boundary_conditions[dir].bc_beg = RADIATION_BC_SYMMETRIC;
                }
                else
                {
                    radiation_data.boundary_conditions[dir].bc_end = RADIATION_BC_SYMMETRIC;
                }

            }
            else if( strcmp( result, "reflective" ) == 0 )
            {
                if( begin )
                {
                    radiation_data.boundary_conditions[dir].bc_beg = RADIATION_BC_REFLECTIVE;
                }
                else
                {
                    radiation_data.boundary_conditions[dir].bc_end = RADIATION_BC_REFLECTIVE;
                }
            }
            else if( strcmp( result, "outflow" ) == 0 )
            {
                if( begin )
                {
                    radiation_data.boundary_conditions[dir].bc_beg = RADIATION_BC_OUTFLOW;
                }
                else
                {
                    radiation_data.boundary_conditions[dir].bc_end = RADIATION_BC_OUTFLOW;
                }
            }
            else if( strcmp( result, "shearingbox" ) == 0 )
            {
                if( begin )
                {
                    radiation_data.boundary_conditions[dir].bc_beg = RADIATION_BC_SHEARINGBOX;
                }
                else
                {
                    radiation_data.boundary_conditions[dir].bc_end = RADIATION_BC_SHEARINGBOX;
                }
            }
            else if( strcmp( result, "fixedvalueE" ) == 0 || strcmp( result, "fixedvalue" ) == 0 )
            {
                if( ParGet( option, 2 ) )
                {
                    if( begin )
                    {
                        radiation_data.boundary_conditions[dir].bc_beg = RADIATION_BC_FIXEDVALUE;
                        radiation_data.boundary_conditions[dir].beg_fixedvalue = atof( ParGet( option, 2 ) ) / ( g_unitDensity * POW2( g_unitVelocity ) );
                    }
                    else
                    {
                        radiation_data.boundary_conditions[dir].bc_end = RADIATION_BC_FIXEDVALUE;
                        radiation_data.boundary_conditions[dir].end_fixedvalue = atof( ParGet( option, 2 ) ) / ( g_unitDensity * POW2( g_unitVelocity ) );
                    }
                }
                else
                {
                    printf( "No value specified for option \"%s fixedvalue <value>\" \n", option );
                    QUIT_PLUTO( 1 );
                }
            }
            else if( strcmp( result, "fixedvalueT" ) == 0 )
            {
                if( ParGet( option, 2 ) )
                {
                    if( begin )
                    {
                        radiation_data.boundary_conditions[dir].bc_beg = RADIATION_BC_FIXEDVALUE;
                        radiation_data.boundary_conditions[dir].beg_fixedvalue =  radiation_data.code_aR * POW4( atof( ParGet( option, 2 ) ) / g_unitTemperature );
                    }
                    else
                    {
                        radiation_data.boundary_conditions[dir].bc_end = RADIATION_BC_FIXEDVALUE;
                        radiation_data.boundary_conditions[dir].end_fixedvalue = radiation_data.code_aR * POW4( atof( ParGet( option, 2 ) ) / g_unitTemperature );
                    }
                }
                else
                {
                    printf( "No value specified for option \"%s fixedvalue <value>\" \n", option );
                    QUIT_PLUTO( 1 );
                }
            }
            else
            {
                printf( "Unknown value for option %s in pluto.ini\n", option );
                QUIT_PLUTO( 1 );
            }

//          result = ParGet(option,1);
//          printf("%s \t %s\n",option,result);
        }
    }

    MPI_Bcast( radiation_data.boundary_conditions, 3 * sizeof( BoundaryConditionInfo ), MPI_CHAR, 0, MPI_COMM_WORLD );

//  for(pos = 0; pos < 3; pos++)
//  {
//      printf("Processor %d \t pos = %d \t bc_beg = %d \t bc_end = %d\n",prank,pos,radiation_data.boundary_conditions[pos].bc_beg,radiation_data.boundary_conditions[pos].bc_end);
//      printf("Processor %d \t pos = %d \t beg_fixedvalue = %e \t end_fixedvalue = %e\n",prank,pos,radiation_data.boundary_conditions[pos].beg_fixedvalue,prank,radiation_data.boundary_conditions[pos].end_fixedvalue);
//  }
    CheckBoundaryConditions( grid );
}

/**
    Checks if the boundary conditions are consistent with the boundary conditions used by
    the HD,... module in PLUTO

    \param[in] grid
*/
void CheckBoundaryConditions( Grid *grid )
{
    int direction = 0;
    int bad = 0;

    for( direction = 0; direction < 3; ++direction ) //iterates over IDIR,JDIR,KDIR
    {
        if( grid[direction].lbound != 0 )
        {
            if( radiation_data.boundary_conditions[direction].bc_beg == RADIATION_BC_PERIODIC && grid[direction].lbound != PERIODIC )
            {
                print( "rX%i-beg set to periodic boundary conditions in radiation module but X%i-beg set to different boundary condition! This is not possible!!!\n", direction + 1, direction + 1 );
                bad = 1;
            }
        }

        if( grid[direction].rbound != 0 )
        {
            if( radiation_data.boundary_conditions[direction].bc_end == RADIATION_BC_PERIODIC && grid[direction].rbound != PERIODIC )
            {
                print( "rX%i-end set to periodic boundary conditions in radiation module but X%i-end set to different boundary condition! This is not possible!!!\n", direction + 1, direction + 1 );
                bad = 1;
            }
        }
    }

    if( bad )
    {
        print( "boundary condition problem\n" );
        QUIT_PLUTO( 1 );
    }
}

/**
    Allocates memory for the array Erad and appends the array to the output system of PLUTO.

    \param[in]  input
*/
void RadiationAllocateMemory( Input *input )
{
    radiation_data.Erad = ( double ** * )Array3D( NX3_TOT, NX2_TOT, NX1_TOT, sizeof( double ) );
    CHECK_ALLOCATED_MEMORY( radiation_data.Erad );

    AppendArrayToPlutoFileOutput( input, "Er", radiation_data.Erad );
}

/**
    Initializes the radiation transport module:

    \note
        RESTART should work by the default mechanism that PLUTO supplies

    \param[in]  argc    PLUTO argc
    \param[in]  argv    PLUTO argv
    \param[in]  input
    \param[in]  grid
    \param[in]  data
    \param[in]  restart

*/
void RadiationInit( int argc, char **argv, Input *input, Grid *grid, Data *data, int restart )
{
    int set_fpe_signal_handler = 0;

    //initialization of some constants in code units
    radiation_data.code_c = CONST_c / g_unitVelocity;
    radiation_data.code_mH = CONST_mH / ( g_unitDensity * POW3( g_unitLength ) );
    radiation_data.code_amu = CONST_mp / ( g_unitDensity * POW3( g_unitLength ) );
    radiation_data.code_aR = CONST_aR / ( ( g_unitDensity * POW2( g_unitVelocity ) ) / POW4( g_unitTemperature ) );
    radiation_data.code_kB = CONST_kB / ( ( g_unitDensity * POW3( g_unitLength ) * POW2( g_unitVelocity ) ) / g_unitTemperature );

    int mat_nrows = grid[KDIR].np_int_glob * grid[JDIR].np_int_glob * grid[IDIR].np_int_glob, nproc = 0;

    if( radiation_data.Erad == NULL )
    {
        RadiationAllocateMemory( input );
    }

//  signal(SIGSEGV, segfault_handle);

    SetupBoundaryConditions( grid );

    if( radiation_data.default_init_Erad )
    {
        if( restart == 0 )
        {
            RadiationSetInitialErad( grid, data );
        }
    }
    RadiationBoundaryCondition( grid, data );

    if( prank == 0 )
    {
        print( "\n> Radiation Settings\n" );
        #if RADIATION_FLUXLIMITER == FLUXLIMITER_MINERBO
            print( " %-25s %s\n", "FLUXLIMITER:", "FLUXLIMITER_MINERBO" );
        #elif RADIATION_FLUXLIMITER == FLUXLIMITER_KLEY
            print( " %-25s %s\n", "FLUXLIMITER:", "FLUXLIMITER_KLEY" );
        #elif RADIATION_FLUXLIMITER == FLUXLIMITER_LEVERMORE_POMRANING
            print( " %-25s %s\n", "FLUXLIMITER:", "FLUXLIMITER_LEVERMORE_POMRANING" );
        #elif RADIATION_FLUXLIMITER == FLUXLIMITER_CONST
            print( " %-25s %s\n", "FLUXLIMITER:", "FLUXLIMITER_CONST" );
        #endif

        #if RADIATION_PRESSURE == YES
            print( " %-25s %s\n", "RADIATION_PRESSURE:", "YES" );
        #else
            print( " %-25s %s\n", "RADIATION_PRESSURE:", "NO" );
        #endif

        #if IRRADIATION == YES
            print( " %-25s %s\n", "IRRADIATION:", "YES" );
        #else
            print( " %-25s %s\n", "IRRADIATION:", "NO" );
        #endif

        #if ARTIFICIAL_DISSIPATION == YES
            print( " %-25s %s\n", "ARTIFICIAL_DISSIPATION:", "YES" );
        #endif

        print( "\n" );

        print( " %-25s %dx%d\n", "Radiation matrix size:", mat_nrows, mat_nrows );
    }

    RadiationPrintUnits();
    PrintBoundaryConditions();

    if( prank == 0 )
    {
        if( ParQuery( "max-iterations" ) )
        {
            radiation_data.max_iterations = atoi( ParGet( "max-iterations", 1 ) );
        }
        else
        {
            radiation_data.max_iterations = 10000;
        }

        if( ParQuery( "absolute-tolerance" ) ) //this is only used for PETSc
        {
            radiation_data.absolute_tolerance = atof( ParGet( "absolute-tolerance", 1 ) );
        }
        else
        {
            radiation_data.absolute_tolerance = 1e-50;
        }

        if( ParQuery( "relative-tolerance" ) )
        {
            radiation_data.relative_tolerance = atof( ParGet( "relative-tolerance", 1 ) );
        }
        else
        {
            radiation_data.relative_tolerance = 1e-5;
        }

        MPI_Bcast( &radiation_data.max_iterations, 1, MPI_INT, 0, MPI_COMM_WORLD );
        MPI_Bcast( &radiation_data.absolute_tolerance, 1, MPI_DOUBLE, 0, MPI_COMM_WORLD );
        MPI_Bcast( &radiation_data.relative_tolerance, 1, MPI_DOUBLE, 0, MPI_COMM_WORLD );
    }
    else
    {
        MPI_Bcast( &radiation_data.max_iterations, 1, MPI_INT, 0, MPI_COMM_WORLD );
        MPI_Bcast( &radiation_data.absolute_tolerance, 1, MPI_DOUBLE, 0, MPI_COMM_WORLD );
        MPI_Bcast( &radiation_data.relative_tolerance, 1, MPI_DOUBLE, 0, MPI_COMM_WORLD );
    }


    #if RADIATION_SOLVER == PETSC_LIBRARY
        PetscInit( argc, argv, input, grid, data );
    #elif RADIATION_SOLVER == SOLVER_SOR
        SuccessiveOverrelaxationInit();
    #endif

    if( ParQuery( "fpe-signal-haldler" ) )
    {
        if( strcmp( ParGet( "fpe-signal-haldler", 1 ), "enable" ) == 0 )
        {
            set_fpe_signal_handler = 1;
        }
    }

    MPI_Bcast( &set_fpe_signal_handler, 1, MPI_INT, 0, MPI_COMM_WORLD );
    if( set_fpe_signal_handler )
    {
        RegisterFpeSignalHandler();
    }

    #if IRRADIATION == YES
        IrradiationInit( grid );
    #endif

    RadiationOpacityInit();

    ShowDomainDecompositionAdvanced( grid );
}

/**
    Cleans up the radiation transport module, frees all allocated memory and calls \ref PetscCleanup()
*/
void RadiationCleanup()
{
    FreeArray3D( ( void * ) radiation_data.Erad );
    radiation_data.Erad = NULL;

    #if RADIATION_SOLVER == PETSC_LIBRARY
        PetscCleanup();
    #elif RADIATION_SOLVER == SOLVER_SOR
        SuccessiveOverrelaxationFinalise();
    #endif

    #if IRRADIATION == YES
        IrradiationFinalise();
    #endif

    SemenovOpacityFinalise();
}

/**
    Computes the next time step

    \param[in]  grid
    \param[in]  data
    \param[in]  input for attribute log_freq

*/
void RadiationImplicitTimestep( Grid *grid, Data *data, Input *input )
{
    int iterations = 0;
    static int min_used_iterations = INT_MAX;
    static int max_used_iterations = 0;
    static int sum_used_iterations = 0;

    #ifdef RADIATION_CUSTOM_IMPLICITE_TEMESTEP_BEGIN
        RadiationCustomImplicitTimestepBegin( grid, data );
    #endif

    #if IRRADIATION == YES
        IrradiationCalculateS( grid, data );
    #endif

    #ifndef RADIATION_DISABLE_MATRIXSOLVER
        #if RADIATION_SOLVER == PETSC_LIBRARY
            iterations = PetscImplicitTimestep( grid, data );
        #elif RADIATION_SOLVER == SOLVER_SOR
            iterations = SuccessiveOverrelaxationImplicitTimestep( grid, data );
        #endif
    #else
            iterations = 0;
    #endif

    #ifdef DEBUG
        int k = 0, j = 0, i = 0;
        TOT_LOOP( k, j, i )
        {
            assert( radiation_data.Erad[k][j][i] >= 0. );
            if( radiation_data.Erad[k][j][i] < 0. )
            {
                print( "ERROR" );
            }
        }
    #endif

    if( prank == 0 )
    {
        if( min_used_iterations > iterations ) min_used_iterations = iterations;
        if( max_used_iterations < iterations ) max_used_iterations = iterations;
        sum_used_iterations += iterations;

        if( input->log_freq == 1 )
        {
            print( "Iterations used for radiation: %d\n", iterations );
        }
        else if( g_stepNumber % input->log_freq == 0 )
        {
            print( "Iterations used for radiation: min=%5d max=%5d average=%5d\n", min_used_iterations, max_used_iterations, sum_used_iterations / input->log_freq );
            min_used_iterations = INT_MAX;
            max_used_iterations = 0;
            sum_used_iterations = 0;
        }
    }

//  if(prank == 0 && g_stepNumber % input->log_freq == 0)
//  {
//      print("Iterations used: %d\n",iterations);
//  }

    #ifdef RADIATION_CUSTOM_IMPLICITE_TEMESTEP_END
        RadiationCustomImplicitTimestepEnd( grid, data, iterations );
    #endif
}

/**
    Computes the next time step by using the SOR (successive over relaxation) algorithm

    \param[in]  grid
    \param[in]  data
*/
int SuccessiveOverrelaxationImplicitTimestep( Grid *grid, Data *data )
{
    int iterations = 0;

    if( ( iterations = SuccessiveOverrelaxation( grid, data, radiation_data.relative_tolerance, radiation_data.absolute_tolerance, radiation_data.max_iterations ) ) == -1 )
    {
       print( "Iterative solver converged\n" );
     
        QUIT_PLUTO( 1 );
    }
    UpdatePlutoPressure( grid, data );
    return iterations;
}

/**
    Allocates all memory needed by the SOR solver

    \note needed for the function \ref SuccessiveOverrelaxation
*/
void SuccessiveOverrelaxationInit()
{
    int i = 0;
    #if RADIATION_SOLVER == SOLVER_SOR
        //Allocate memory for SOLVER SOR

        if( prank == 0 )
        {
            print( ">Radiation Solver Settings\n" );
            print( " %-25s %s\n", "SOLVER:", "SOR" );
            #if RADIATION_SOR_OMEGA_MODE == SOR_ADAPTIVE_OMEGA
                print( " %-25s %s\n", "SOR_OMEGA_MODE:", "SOR_ADAPTIVE_OMEGA" );
            #elif RADIATION_SOR_OMEGA_MODE == SOR_FIXED_OMEGA
                print( " %-25s %s\n", "SOR_OMEGA_MODE:", "SOR_FIXED_OMEGA" );
            #endif

            #if RADIATION_SOR_NORM == NORM_RELATIVE_L2
                print( " %-25s %s\n", "NORM:", "NORM_RELATIVE_L2" );
            #elif RADIATION_SOR_NORM == NORM_RELATIVE_MAX
                print( " %-25s %s\n", "NORM:", "NORM_RELATIVE_MAX" );
            #elif RADIATION_SOR_NORM == NORM_RESIDUAL_L2
                print( " %-25s %s\n", "NORM:", "NORM_RESIDUAL_L2" );
            #elif RADIATION_SOR_NORM == NORM_RESIDUAL_MAX
                print( " %-25s %s\n", "NORM:", "NORM_RESIDUAL_MAX" );
            #elif RADIATION_SOR_NORM == NORM_RESIDUAL_RELATIVE_L2
                print( " %-25s %s\n", "NORM:", "NORM_RESIDUAL_RELATIVE_L2" );
            #elif RADIATION_SOR_NORM == NORM_RESIDUAL_RELATIVE_MAX
                print( " %-25s %s\n", "NORM:", "NORM_RESIDUAL_RELATIVE_MAX" );
            #elif RADIATION_SOR_NORM == NORM_PETSC
                print( " %-25s %s\n", "NORM:", "NORM_PETSC" );
            #endif

            print( " %-25s %5.3e\n", "absolute-tolerance:", radiation_data.absolute_tolerance );
            print( " %-25s %5.3e\n", "relative-tolerance:" , radiation_data.relative_tolerance );
            print( " %-25s %d\n", "max-iterations:", radiation_data.max_iterations );
            print( "\n" );
        }

        radiation_data.sor_coefficients = ( double ** ** )Array4D( 8, NX3_TOT, NX2_TOT, NX1_TOT, sizeof( double ) );
        CHECK_ALLOCATED_MEMORY( radiation_data.sor_coefficients );
    #endif
}

/**
    Frees the memory allocated by function \ref SuccessiveOverrelaxationInit

    \note needed for the function \ref SuccessiveOverrelaxation
*/
void SuccessiveOverrelaxationFinalise()
{
    #if RADIATION_SOLVER == SOLVER_SOR
        FreeArray4D( ( void ** ** )radiation_data.sor_coefficients );
        radiation_data.sor_coefficients = NULL;
    #endif
}

/**
    Computes the new omega for the successive over relaxation \ref SuccessiveOverrelaxation. This function is used
    if RADIATION_SOR_OMEGA_MODE is set to SOR_ADAPTIVE_OMEGA otherwise omega is set to a constant value. Omega
    is computed by using the averaged number of iteration used by the function \ref SuccessiveOverrelaxation. It is
    averaged over \a averageing_timesteps time steps and omega is increased or decreased by the value of \a increase.
    The omega is only updated every \a averageing_timesteps time steps.

    \param[in]  omega
    \param[in]  iterations
    \param[in]  increase
    \param[in]  averageing_timesteps

    \returns the computed omega
*/
double ComputeAdaptiveOmega( double omega, int iterations, double increase, int averageing_timesteps )
{
    static double current_max_iterations = 0;
    static double current_average_iteratios = 0;
    static double last_average_iterations = 0.;
    static double direction = 1.0;
    double diff = 0.0;

    current_average_iteratios += ( double )iterations;
    current_max_iterations = MAX( current_max_iterations, ( double )iterations );

    if( g_stepNumber % averageing_timesteps == 0 )
    {
        current_average_iteratios = current_average_iteratios / ( ( double )averageing_timesteps );

        if( current_max_iterations > current_average_iteratios * 3.0 )
        {
            current_average_iteratios = current_max_iterations;
        }

        diff = current_average_iteratios - last_average_iterations;

        if( diff > 0. )
        {
            omega = omega + ( -1.0 * direction ) * increase /* boost*/;
            direction = -1.0 * direction;
        }
        else if( diff < 0. )
        {
            omega = omega + direction * increase /* boost*/;
        }

        omega = omega < 0.1 ? 0.1 : omega;
        omega = omega > 1.95 ? 1.95 : omega;

//      if(prank == 0 && g_stepNumber % 100 == 0)
//      {
//              print("current_average_iteratios=%10.15e last_average_iterations=%10.15e direction=%10.15e omega=%10.15e diff=%10.15e\n",current_average_iteratios,last_average_iterations,direction,omega,diff);
//      }

        last_average_iterations = current_average_iteratios;
        current_max_iterations = 0.;
        current_average_iteratios = 0.;

        return omega;
    }
    else
    {
        return omega;
    }
}

/**
    Computes the new radiation energy density by using the iterative matrix solver
    SOR (successive over relaxation).

    Features:\n
        Adaptive change of \a omega during the simulation: To activate set RADIATION_SOR_OMEGA_MODE = SOR_ADAPTIVE_OMEGA
        Different variants to compute the norm for the stop criterion: NORM_RELATIVE_L2, NORM_RELATIVE_MAX, (NORM_RESIDUAL_L2, NORM_RESIDUAL_MAX)

    \note
        If adaptive omega calculation is used (RADIATION_SOR_OMEGA_MODE is set to SOR_ADAPTIVE_OMEGA) we do not finally stop if max_iterations is reached or if norm_change < 1e-20.
        Instead we change omega to an different value and try to solve it again.


    Stop criterion:\n
        The computation is stopped if norm <= rtol or then then iterations > \a max_iterations. \n \n

        The norm \f$ n \f$ can be computed in different ways:\n
        if RADIATION_SOR_NORM == NORM_RELATIVE_L2
        \f[
            n = \frac{\left\| \vec{x}^{(k)} - \vec{x}^{(k-1)}\right\|}{\left\| \vec{x}^{(k)}\right\|}
        \f]
        if RADIATION_SOR_NORM == NORM_RELATIVE_MAX
        \f[
            n = \frac{\left\| \vec{x}^{(k)} - \vec{x}^{(k-1)}\right\|_{\infty}}{\left\| \vec{x}^{(k)}\right\|_{\infty}}
        \f]
        if RADIATION_SOR_NORM == NORM_RESIDUAL_L2
        \f[
            n = \frac{\left\| \vec{r}^{(k)}\right\|}{\left\| \vec{b}^{(k)}\right\|}
        \f]
        if RADIATION_SOR_NORM == NORM_RESIDUAL_MAX
        \f[
            n = \frac{\left\| \vec{r}^{(k)}\right\|_{\infty}}{\left\| \vec{b}^{(k)}\right\|_{\infty}}
        \f]
        if RADIATION_SOR_NORM == NORM_RESIDUAL_RELATIVE_L2
        \f[
            n = \frac{\left\| \vec{r}^{(k)}\right\|}{\left\| \vec{r}^{(0)}\right\|}
        \f]
        if RADIATION_SOR_NORM == NORM_RESIDUAL_RELATIVE_MAX
        \f[
            n = \frac{\left\| \vec{r}^{(k)}\right\|_{\infty}}{\left\| \vec{r}^{(0)}\right\|_{\infty}}
        \f]
        if RADIATION_SOR_NORM == NORM_PETSC
        The iteration stops if the expression
        \f[
            \left\| \vec{r}^{(k)} \right\| < \max(rtol \cdot \left\| \vec{b} \right\|, atol)
        \f]
        is true. The relative tolerance is rtol and the absolute tolerance atol.

        In the above formulas is \f$ \vec{x}^{(k)} \f$ the solution vector (the radiation energy density) at the k-th iteration of the matrix equation \f$ A \vec{x} =\vec{b} \f$ and
        \f$ \vec{r}^{(k)} \f$ is the residual at the k-th iteration which is given by the expression \f$ \vec{r}^{(k)} = \vec{b} - A \vec{x}^{(k)}\f$. The norm \f$ \left\| \vec{x}\right\| \f$
        is computed with the formula
        \f[
            \left\| \vec{x}\right\| = \sqrt{\sum_i x_i^2}
        \f]
        and \f$ \left\| \vec{x}\right\|_{\infty} \f$ by
        \f[
            \left\| \vec{x}\right\|_{\infty} = \max_{i} x_i
        \f]

    \param[in]  grid
    \param[in]  data
    \param[in]  rtol            relative tolerance
    \param[in]  atol            absolute tolerance
    \param[in]  max_iterations
    \param[out] data
*/
int SuccessiveOverrelaxation( Grid *grid, Data *data, double rtol, double atol, int max_iterations )
{
    #if RADIATION_SOLVER == SOLVER_SOR
        int i = 0, j = 0, k = 0;
        double *** Erad = radiation_data.Erad;
        double norm = 0., norm_change = 0., tmp = 0.;
        static double omega = 1.;
        int iterations = 0;

        #if RADIATION_SOR_OMEGA_MODE == SOR_ADAPTIVE_OMEGA
            int outer_iterations = 0;
        #endif

        double norm_tmp[2];
        norm_tmp[0] = 0.0;
        norm_tmp[1] = 0.0;

        DOM_LOOP( k, j, i )
        {
    //      radiation_data.sor_coefficients[0][k][j][i] = ComputeCoefficientU1(k,j,i,data,grid);
            radiation_data.sor_coefficients[1][k][j][i] = ComputeCoefficientU2( k, j, i, data, grid );
            radiation_data.sor_coefficients[2][k][j][i] = ComputeCoefficientU3( k, j, i, data, grid );
            radiation_data.sor_coefficients[3][k][j][i] = ComputeCoefficientU4( k, j, i, data, grid );
            radiation_data.sor_coefficients[4][k][j][i] = ComputeCoefficientU5( k, j, i, data, grid );
            radiation_data.sor_coefficients[5][k][j][i] = ComputeCoefficientU6( k, j, i, data, grid );
            radiation_data.sor_coefficients[6][k][j][i] = ComputeCoefficientU7( k, j, i, data, grid );
            radiation_data.sor_coefficients[7][k][j][i] = ComputeCoefficientB( k, j, i, data, grid );

            radiation_data.sor_coefficients[0][k][j][i] = ComputeCoefficientU1Advanced( k, j, i, data, grid, radiation_data.sor_coefficients[1][k][j][i], radiation_data.sor_coefficients[2][k][j][i], radiation_data.sor_coefficients[3][k][j][i], radiation_data.sor_coefficients[4][k][j][i], radiation_data.sor_coefficients[5][k][j][i], radiation_data.sor_coefficients[6][k][j][i] );


            #if RADIATION_SOR_NORM == NORM_RESIDUAL_L2 || RADIATION_SOR_NORM == NORM_PETSC
                    norm_tmp[1] += POW2( radiation_data.sor_coefficients[7][k][j][i] );
            #elif RADIATION_SOR_NORM == NORM_RESIDUAL_MAX
                    norm_tmp[1] = fabs( MAX( norm_tmp[1], radiation_data.sor_coefficients[7][k][j][i] ) );
            #endif
        }
        #if RADIATION_SOR_NORM == NORM_RESIDUAL_L2 || RADIATION_SOR_NORM == NORM_PETSC
            MPI_Allreduce( &norm_tmp[1], &tmp, 1, MPI_DOUBLE, MPI_SUM, MPI_COMM_WORLD );
            norm_tmp[1] = sqrt( tmp );
        #elif RADIATION_SOR_NORM == NORM_RESIDUAL_MAX
            MPI_Allreduce( &norm_tmp[1], &tmp, 1, MPI_MAX, MPI_MAX, MPI_COMM_WORLD );
            norm_tmp[1] = tmp;
        #endif

        #if RADIATION_SOR_OMEGA_MODE == SOR_ADAPTIVE_OMEGA
            do
            {
                if( outer_iterations == 1 )
                {
                    omega = 1.0;
                }
                else if( outer_iterations == 2 )
                {
                    omega = 0.9;
                }
                else if( outer_iterations > 2 )
                {
                    break;
                }

                outer_iterations++;
        #endif
        norm = 2.0 * rtol;
        norm_change = norm;
        iterations = 0;

        #if RADIATION_SOR_NORM == NORM_PETSC
            norm = 10.0 * rtol * norm_tmp[1];
            while( norm > MAX( rtol * norm_tmp[1], atol ) && norm_change > 1e-20 && max_iterations > iterations )
        #else
            while( norm > rtol && norm_change > 1e-20 && max_iterations > iterations )
        #endif
        {
            RadiationBoundaryCondition( grid, data );
            norm_change = norm;
            norm = 0;
            norm_tmp[0] = 0.0;
            #if RADIATION_SOR_NORM != NORM_RESIDUAL_L2 && RADIATION_SOR_NORM != NORM_RESIDUAL_MAX && RADIATION_SOR_NORM != NORM_RESIDUAL_RELATIVE_L2 && RADIATION_SOR_NORM != NORM_RESIDUAL_RELATIVE_MAX && RADIATION_SOR_NORM != NORM_PETSC
                norm_tmp[1] = 0.0;
            #endif

            DOM_LOOP( k, j, i )
            {
                #if RADIATION_SOR_NORM == NORM_RELATIVE_MAX || RADIATION_SOR_NORM == NORM_RELATIVE_L2
                    tmp = Erad[k][j][i];
                #endif

                Erad[k][j][i] = ( 1. - omega ) * Erad[k][j][i] - ( omega / radiation_data.sor_coefficients[0][k][j][i] ) * ( radiation_data.sor_coefficients[1][k][j][i] * Erad[k - 1][j][i] + radiation_data.sor_coefficients[2][k][j][i] * Erad[k + 1][j][i] + radiation_data.sor_coefficients[3][k][j][i] * Erad[k][j - 1][i] + radiation_data.sor_coefficients[4][k][j][i] * Erad[k][j + 1][i] + radiation_data.sor_coefficients[5][k][j][i] * Erad[k][j][i - 1] + radiation_data.sor_coefficients[6][k][j][i] * Erad[k][j][i + 1] - radiation_data.sor_coefficients[7][k][j][i] );

                #if RADIATION_SOR_NORM == NORM_RELATIVE_L2
                    norm_tmp[0] += POW2( tmp - Erad[k][j][i] ) / POW2( tmp );
                #elif RADIATION_SOR_NORM == NORM_RELATIVE_MAX
                    norm_tmp[0] = fabs( MAX( norm_tmp[0], POW2( tmp - Erad[k][j][i] ) ) );
                    norm_tmp[1] = fabs( MAX( norm_tmp[1], POW2( tmp ) ) );
                #endif
            }

            #if RADIATION_SOR_NORM == NORM_RESIDUAL_MAX || RADIATION_SOR_NORM == NORM_RESIDUAL_L2 || RADIATION_SOR_NORM == NORM_RESIDUAL_RELATIVE_MAX || RADIATION_SOR_NORM == NORM_RESIDUAL_RELATIVE_L2 || RADIATION_SOR_NORM == NORM_PETSC
                norm_tmp[0] = 0.0;
                DOM_LOOP( k, j, i )
                {
                    ///\f$ \vec{r}^{k} = \vec{b} - A \vec{x}^{k} \f$
                    tmp = radiation_data.sor_coefficients[7][k][j][i] - ( radiation_data.sor_coefficients[0][k][j][i] * Erad[k][j][i] + radiation_data.sor_coefficients[1][k][j][i] * Erad[k - 1][j][i] + radiation_data.sor_coefficients[2][k][j][i] * Erad[k + 1][j][i] + radiation_data.sor_coefficients[3][k][j][i] * Erad[k][j - 1][i] + radiation_data.sor_coefficients[4][k][j][i] * Erad[k][j + 1][i] + radiation_data.sor_coefficients[5][k][j][i] * Erad[k][j][i - 1] + radiation_data.sor_coefficients[6][k][j][i] * Erad[k][j][i + 1] );
                    #if RADIATION_SOR_NORM == NORM_RESIDUAL_L2 || RADIATION_SOR_NORM == NORM_RESIDUAL_RELATIVE_L2 || RADIATION_SOR_NORM == NORM_PETSC
                        norm_tmp[0] += POW2( tmp );
                    #elif RADIATION_SOR_NORM == NORM_RESIDUAL_MAX || RADIATION_SOR_NORM == NORM_RESIDUAL_RELATIVE_MAX
                        norm_tmp[0] = fabs( MAX( norm_tmp[0], tmp ) );
                    #endif
                }
            #endif

                ///\note the distribution of norm asures that each processor makes the same number of iterations so it is not necessary to distribute the number of iterations
            #if RADIATION_SOR_NORM == NORM_RELATIVE_L2
                MPI_Allreduce( &norm_tmp[0], &norm, 1, MPI_DOUBLE, MPI_SUM, MPI_COMM_WORLD );
                norm = sqrt( norm );
            #elif RADIATION_SOR_NORM == NORM_RESIDUAL_RELATIVE_L2
                MPI_Allreduce( &norm_tmp[0], &tmp, 1, MPI_DOUBLE, MPI_SUM, MPI_COMM_WORLD );
                norm_tmp[0] = tmp;
                if( iterations == 0 )
                {
                    norm_tmp[1] = norm_tmp[0];
                    norm = 4.0 * rtol;
                }
                else
                {
                    norm = sqrt( norm_tmp[0] ) / sqrt( norm_tmp[1] );
                }
            #elif RADIATION_SOR_NORM == NORM_PETSC
                MPI_Allreduce( &norm_tmp[0], &tmp, 1, MPI_DOUBLE, MPI_SUM, MPI_COMM_WORLD );
                norm = norm_tmp[0] = sqrt( tmp );
            #elif RADIATION_SOR_NORM == NORM_RESIDUAL_L2
                MPI_Allreduce( &norm_tmp[0], &tmp, 1, MPI_DOUBLE, MPI_SUM, MPI_COMM_WORLD );
                norm_tmp[0] = tmp;
                norm = sqrt( norm_tmp[0] ) / norm_tmp[1];
            #elif RADIATION_SOR_NORM == NORM_RELATIVE_MAX
                MPI_Allreduce( &norm_tmp[0], &tmp, 1, MPI_DOUBLE, MPI_MAX, MPI_COMM_WORLD );
                norm_tmp[0] = tmp;
                MPI_Allreduce( &norm_tmp[1], &tmp, 1, MPI_DOUBLE, MPI_MAX, MPI_COMM_WORLD );
                norm_tmp[1] = tmp;
                norm = norm_tmp[0] / norm_tmp[1];
            #elif RADIATION_SOR_NORM == NORM_RESIDUAL_RELATIVE_MAX
                MPI_Allreduce( &norm_tmp[0], &tmp, 1, MPI_DOUBLE, MPI_MAX, MPI_COMM_WORLD );
                norm_tmp[0] = tmp;
                norm = norm_tmp[0] / norm_tmp[1];
                if( iterations == 0 )
                {
                    norm_tmp[1] = norm_tmp[0];
                    norm = 4.0 * rtol;
                }
                else
                {
                    norm = norm_tmp[0] / norm_tmp[1];
                }
            #elif RADIATION_SOR_NORM == NORM_RESIDUAL_MAX
                MPI_Allreduce( &norm_tmp[0], &tmp, 1, MPI_DOUBLE, MPI_MAX, MPI_COMM_WORLD );
                norm_tmp[0] = tmp;
                norm = norm_tmp[0] / norm_tmp[1];
            #endif

            if( !isnormal( norm ) ) return -1;
            if( !isnormal( norm_change ) ) return -1;

            norm_change = fabs( norm - norm_change );
//             if(prank == 0)
//             {
//                 print("SOR iteration=%5d norm=%15.10e rtol=%15.10e\n",iterations,norm,rtol);
//             }

            iterations++;
        }
        #if RADIATION_SOR_OMEGA_MODE == SOR_ADAPTIVE_OMEGA
            }
            while( norm_change <= 1e-20 && max_iterations <= iterations );
        #endif

        if( max_iterations == iterations || norm_change < 1e-20 )
        {
            if( max_iterations == iterations ) print( "Maximum number of iterations reached!\n" );
            return -1;
        }

        #if RADIATION_SOR_OMEGA_MODE == SOR_ADAPTIVE_OMEGA
//             omega = ComputeAdaptiveOmega(omega,iterations - 1, 0.005, 20);
            omega = ComputeAdaptiveOmega( omega, iterations - 1, 0.0005, 100 );
        #endif

        RadiationBoundaryCondition( grid, data );
        return iterations - 1;
    #else
        return -1;
    #endif
}

/**
    Calculates the flux limiter

    \param[in] R

    \returns
        if RADIATION_FLUXLIMITER is set to FLUXLIMITER_MINERBO then
        \f[
        \lambda(R) = \left\{
            \begin{array}{l l}
                \frac{2}{3 + \sqrt{9+12 R^2}} & \quad 0 \leq R \leq \frac{3}{2}\\
                \frac{1}{1+R+\sqrt{1+2R}} & \quad \frac{3}{2} < R \leq \infty\\
            \end{array} \right.\\
        \f]
        is returned \n
        if RADIATION_FLUXLIMITER is set to FLUXLIMITER_KLEY then
        \f[
        \lambda(R) = \left\{
            \begin{array}{l l}
                \frac{2}{3 + \sqrt{9+10 R^2}} & \quad 0 \leq R \leq 2\\
                \frac{10}{10 R+ 9 + \sqrt{180 R + 81}} & \quad 2 < R \leq \infty\\
            \end{array} \right.\\
        \f]
        is returned \n
        if RADIATION_FLUXLIMITER is set to FLUXLIMITER_LEVERMORE_POMRANING then
        \f[
        \lambda(R) = \frac{1}{R} \left( \coth R - \frac{1}{R}\right)
        \f]
        is returned \n
        if RADIATION_FLUXLIMITER is set to FLUXLIMITER_CONST then
        \f[
        \frac{1}{3}
        \f]
        is returned

*/
double Fluxlimiter( double R )
{
    assert( R >= 0. );

    #if RADIATION_FLUXLIMITER == FLUXLIMITER_MINERBO

        if( R <= 3. / 2. ) // 0 <= R <= 3./2.
        {
            return 2. / ( 3. + sqrt( 9. + 12. * R * R ) );
        }
        else // 3./2. < R <= /infty
        {
            return 1. / ( 1. + R + sqrt( 1. + 2. * R ) );
        }
    #elif RADIATION_FLUXLIMITER == FLUXLIMITER_KLEY
    //from dissertation kley
        if( R <= 2. ) // 0 <= R <= 2.
        {
            return 2. / ( 3. + sqrt( 9. + 10. * R * R ) );
        }
        else // 2. < R <= /infty
        {
            return 10. / ( 10.* R + 9. + sqrt( 180. * R + 81. ) );
        }
    #elif RADIATION_FLUXLIMITER == FLUXLIMITER_LEVERMORE_POMRANING
        return ( 1. / R ) * ( cosh( R ) / sinh( R ) - ( 1. / R ) );
    #elif RADIATION_FLUXLIMITER == FLUXLIMITER_CONST
        return 1. / 3.;
    #else
        #error RADIATION_FLUXLIMITER not set!
    #endif
}

/**
    Computes the intermediate value of a given variable.

    \par Shift
        Here the meaning of the parameter shift is explained. In the following we expect that \a variable is RHO (\f$ \rho \f$) \n
        if shift == \ref NO_SHIFT
            \f[
                \rho_{i,j,k}
            \f]
        is returned \n
        if shift == \ref IDIR_pOH
            \f[
                \rho_{i+\frac{1}{2},j,k} \approx \frac{\rho_{i,j,k} + \rho_{i+1,j,k}}{2}
            \f]
        is returned \n
        if shift == \ref IDIR_mOH
            \f[
                \rho_{i-\frac{1}{2},j,k} \approx \frac{\rho_{i-1,j,k} + \rho_{i,j,k}}{2}
            \f]
        is returned \n
        if shift == \ref JDIR_pOH
            \f[
                \rho_{i,j+\frac{1}{2},k} \approx \frac{\rho_{i,j,k} + \rho_{i,j+1,k}}{2}
            \f]
        is returned \n
        if shift == \ref JDIR_mOH
            \f[
                \rho_{i,j-\frac{1}{2},k} \approx \frac{\rho_{i,j-1,k} + \rho_{i,j,k}}{2}
            \f]
        is returned \n
        if shift == \ref KDIR_pOH
            \f[
                \rho_{i,j,k+\frac{1}{2}} \approx \frac{\rho_{i,j,k} + \rho_{i,j,k+1}}{2}
            \f]
        is returned \n
        if shift == \ref KDIR_mOH
            \f[
                \rho_{i,j,k-\frac{1}{2}} \approx \frac{\rho_{i,j,k-1} + \rho_{i,j,k}}{2}
            \f]
        is returned

    \note If you change the way of averaging in this function then you hav also modify it in function \ref ComputeIntermediateValueWithFunction()

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  shift   is explained above
    \param[in]  quantity    array for which the intermediate value is computed

    \returns        intermediate value
*/
double ComputeIntermediateValue( int k, int j, int i, int shift, double ***quantity )
{
    switch( shift )
    {
        case NO_SHIFT:
            return quantity[k][j][i];
            break;

        case IDIR_pOH:
            return ( quantity[k][j][i] + quantity[k][j][i + 1] ) / 2.;
            break;

        case IDIR_mOH:
            return ( quantity[k][j][i - 1] + quantity[k][j][i] ) / 2.;
            break;

        case JDIR_pOH:
            return ( quantity[k][j][i] + quantity[k][j + 1][i] ) / 2.;
            break;

        case JDIR_mOH:
            return ( quantity[k][j - 1][i] + quantity[k][j][i] ) / 2.;
            break;

        case KDIR_pOH:
            return ( quantity[k][j][i] + quantity[k + 1][j][i] ) / 2.;
            break;

        case KDIR_mOH:
            return ( quantity[k - 1][j][i] + quantity[k][j][i] ) / 2.;
            break;

        default:
            print( "Unexpected value of shift\n" );
            QUIT_PLUTO( 1 );
            break;
    }
    return 0.0; //should never be reached
}

/**
    Computes the intermediate value of a variable generated by function \a getVariable. For further details see \ref ComputeIntermediateValue

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  shift
    \param[in]  data
    \param[in]  getVariable function pointer to access variables like the temperature which is computed from the density and the pressure
    \returns        intermediate value
*/
double ComputeIntermediateValueWithFunction( int k, int j, int i, int shift, Data *data, double( getVariable )( int, int, int, Data * ) )
{
    switch( shift )
    {
        case NO_SHIFT:
            return getVariable( k, j, i, data );
            break;

        case IDIR_pOH:
            return ( getVariable( k, j, i, data ) + getVariable( k, j, i + 1, data ) ) / 2.;
            break;

        case IDIR_mOH:
            return ( getVariable( k, j, i - 1, data ) + getVariable( k, j, i, data ) ) / 2.;
            break;

        case JDIR_pOH:
            return ( getVariable( k, j, i, data ) + getVariable( k, j + 1, i, data ) ) / 2.;
            break;

        case JDIR_mOH:
            return ( getVariable( k, j - 1, i, data ) + getVariable( k, j, i, data ) ) / 2.;
            break;

        case KDIR_pOH:
            return ( getVariable( k, j, i, data ) + getVariable( k + 1, j, i, data ) ) / 2.;
            break;

        case KDIR_mOH:
            return ( getVariable( k - 1, j, i, data ) + getVariable( k, j, i, data ) ) / 2.;
            break;

        default:
            print( "Unexpected value of shift\n" );
            QUIT_PLUTO( 1 );
            break;
    }
    return 0.0; //sould never be reached
}

/**
    This function calculates \f$ R = \frac{\left|\nabla E \right|}{\kappa_R E} \f$ at different positions
    depending on the value of \a shift:

    shift == NO_SHIFT
        \f[ R_{i,j,k} = \frac{\left|(\nabla E)_{i,j,k} \right|}{{\kappa_R}_{i,j,k} \rho_{i,j,k} E_{i,j,k}} \f]
    shift == IDIR_pOH
        \f[ R_{i+\frac{1}{2},j,k} = \frac{\left|(\nabla E)_{i+\frac{1}{2},j,k} \right|}{{\kappa_R}_{i+\frac{1}{2},j,k} \rho_{i+\frac{1}{2},j,k} E_{i+\frac{1}{2},j,k}} \f]
    shift == IDIR_mOH
        \f[ R_{i-\frac{1}{2},j,k} = \frac{\left|(\nabla E)_{i-\frac{1}{2},j,k} \right|}{{\kappa_R}_{i-\frac{1}{2},j,k} \rho_{i-\frac{1}{2},j,k} E_{i-\frac{1}{2},j,k}} \f]

    and analog for shift == JDIR_pOH,JDIR_mOH,KDIR_pOH,KDIR_mOH

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  shift is explained above
    \param[in]  d
    \param[in]  grid

    \returns \f$ R = \frac{\left|\nabla E \right|}{\kappa_R \rho E} \f$ if shift == NO_SHIFT else wise \f$ R \f$ accordingly to the value of shift see above
*/
double FluxlimiterR( int k, int j, int i, int shift, Data *d, Grid *grid )
{
    double abs_grad_Erad = 0.;
    double numerator_x1 = 0., numerator_x2 = 0., numerator_x3 = 0., denominator = 0.;

    switch( shift )
    {
        case NO_SHIFT:
            numerator_x1 = ComputeIntermediateValue( k, j, i, IDIR_pOH, radiation_data.Erad ) - ComputeIntermediateValue( k, j, i, IDIR_mOH, radiation_data.Erad );
            numerator_x2 = ComputeIntermediateValue( k, j, i, JDIR_pOH, radiation_data.Erad ) - ComputeIntermediateValue( k, j, i, JDIR_mOH, radiation_data.Erad );
            numerator_x3 = ComputeIntermediateValue( k, j, i, KDIR_pOH, radiation_data.Erad ) - ComputeIntermediateValue( k, j, i, KDIR_mOH, radiation_data.Erad );
            break;

        case IDIR_pOH:
            numerator_x1 = radiation_data.Erad[k][j][i + 1] - radiation_data.Erad[k][j][i];
            numerator_x2 = ComputeIntermediateValue( k, j, i, JDIR_pOH, radiation_data.Erad ) - ComputeIntermediateValue( k, j, i, JDIR_mOH, radiation_data.Erad );
            numerator_x3 = ComputeIntermediateValue( k, j, i, KDIR_pOH, radiation_data.Erad ) - ComputeIntermediateValue( k, j, i, KDIR_mOH, radiation_data.Erad );
            break;

        case IDIR_mOH:
            numerator_x1 = radiation_data.Erad[k][j][i] - radiation_data.Erad[k][j][i - 1];
            numerator_x2 = ComputeIntermediateValue( k, j, i, JDIR_pOH, radiation_data.Erad ) - ComputeIntermediateValue( k, j, i, JDIR_mOH, radiation_data.Erad );
            numerator_x3 = ComputeIntermediateValue( k, j, i, KDIR_pOH, radiation_data.Erad ) - ComputeIntermediateValue( k, j, i, KDIR_mOH, radiation_data.Erad );
            break;

        case JDIR_pOH:
            numerator_x1 = ComputeIntermediateValue( k, j, i, IDIR_pOH, radiation_data.Erad ) - ComputeIntermediateValue( k, j, i, IDIR_mOH, radiation_data.Erad );
            numerator_x2 = radiation_data.Erad[k][j + 1][i] - radiation_data.Erad[k][j][i];
            numerator_x3 = ComputeIntermediateValue( k, j, i, KDIR_pOH, radiation_data.Erad ) - ComputeIntermediateValue( k, j, i, KDIR_mOH, radiation_data.Erad );
            break;

        case JDIR_mOH:
            numerator_x1 = ComputeIntermediateValue( k, j, i, IDIR_pOH, radiation_data.Erad ) - ComputeIntermediateValue( k, j, i, IDIR_mOH, radiation_data.Erad );
            numerator_x2 = radiation_data.Erad[k][j][i] - radiation_data.Erad[k][j - 1][i];
            numerator_x3 = ComputeIntermediateValue( k, j, i, KDIR_pOH, radiation_data.Erad ) - ComputeIntermediateValue( k, j, i, KDIR_mOH, radiation_data.Erad );
            break;

        case KDIR_pOH:
            numerator_x1 = ComputeIntermediateValue( k, j, i, IDIR_pOH, radiation_data.Erad ) - ComputeIntermediateValue( k, j, i, IDIR_mOH, radiation_data.Erad );
            numerator_x2 = ComputeIntermediateValue( k, j, i, JDIR_pOH, radiation_data.Erad ) - ComputeIntermediateValue( k, j, i, JDIR_mOH, radiation_data.Erad );
            numerator_x3 = radiation_data.Erad[k + 1][j][i] - radiation_data.Erad[k][j][i];
            break;

        case KDIR_mOH:
            numerator_x1 = ComputeIntermediateValue( k, j, i, IDIR_pOH, radiation_data.Erad ) - ComputeIntermediateValue( k, j, i, IDIR_mOH, radiation_data.Erad );
            numerator_x2 = ComputeIntermediateValue( k, j, i, JDIR_pOH, radiation_data.Erad ) - ComputeIntermediateValue( k, j, i, JDIR_mOH, radiation_data.Erad );
            numerator_x3 = radiation_data.Erad[k][j][i] - radiation_data.Erad[k - 1][j][i];
            break;

        default:
            print( "Unexpected value of shift\n" );
            QUIT_PLUTO( 1 );
            break;
    }

    #if (GEOMETRY == CARTESIAN)
        abs_grad_Erad = sqrt(
                            POW2( numerator_x1 / ( grid[IDIR].dx[i] ) )
                            + POW2( numerator_x2 / ( grid[JDIR].dx[j] ) )
                            + POW2( numerator_x3 / ( grid[KDIR].dx[k] ) )
                        );
    #elif (GEOMETRY == POLAR)
        abs_grad_Erad = sqrt(
                            POW2( numerator_x1 / ( grid[IDIR].dx[i] ) )
                            + POW2( numerator_x2 / ( grid[JDIR].dx[j] * grid[IDIR].x[i] ) )
                            + POW2( numerator_x3 / ( grid[KDIR].dx[k] ) )
                        );
    #elif (GEOMETRY == CYLINDRICAL)
        #ERROR CYLINDRICAL not implemented so far
    #elif (GEOMETRY == SPHERICAL)
        abs_grad_Erad = sqrt(
                            POW2( numerator_x1 / ( grid[IDIR].dx[i] ) )
                            + POW2( numerator_x2 / ( grid[JDIR].dx[j] * grid[IDIR].x[i] ) )
                            + POW2( numerator_x3 / ( grid[KDIR].dx[k] * grid[IDIR].x[i] * sin( grid[JDIR].x[j] ) ) )
                        );
    #endif

    #ifdef DEBUG
        double rho_im = ComputeIntermediateValue( k, j, i, shift, d->Vc[RHO] );
        double E_im = ComputeIntermediateValue( k, j, i, shift, radiation_data.Erad );
        assert( E_im > 0. );
        denominator = ComputeRosselandOpacity( k, j, i, shift, d, grid ) * rho_im * E_im;
    #else
        #if RADIATION_SMALL_NUMBERS_FIX == YES
            denominator = ComputeRosselandOpacity( k, j, i, shift, d, grid ) * ComputeIntermediateValue( k, j, i, shift, d->Vc[RHO] ) * ComputeIntermediateValue( k, j, i, shift, radiation_data.Erad ) + RADIATION_SMALL_NUMBERS_FIX_VALUE;
        #else
            denominator = ComputeRosselandOpacity( k, j, i, shift, d, grid ) * ComputeIntermediateValue( k, j, i, shift, d->Vc[RHO] ) * ComputeIntermediateValue( k, j, i, shift, radiation_data.Erad );
        #endif
    #endif

    assert( ( abs_grad_Erad / denominator ) >= 0. );

    return abs_grad_Erad / denominator;
}

/**
    Returns the Rosseland mean opacity in code units. The cgs-units of the opacity is \f$ [\kappa_R] = \frac{cm^2}{g}\f$

    \note For informations about the parameter \a shift look at function \ref ComputeIntermediateValue

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  shift
    \param[in]  d
    \param[in]  grid

    \returns Rosseland mean opacity in code units

*/
double ComputeRosselandOpacity( int k, int j, int i, int shift, Data *d, Grid *grid )
{   
    // check pressione negativa
    double pres_aux = d->Vc[PRS][k][j][i];
    if (pres_aux < 0.0) pres_aux = g_smallPressure;
 
    double conv = 11604.525;
    double cgs_density = ComputeIntermediateValue( k, j, i, shift, d->Vc[RHO])* g_unitDensity;   /**<density in log cgs units */
    double cgs_temperature = ComputeIntermediateValueWithFunction( k, j, i, shift, d, GetTemperature )*g_unitTemperature ; /**<temperature in log cgs units */
    double result;

    assert( cgs_density > 0. );
    assert( cgs_temperature > 0. );

    result = RadiationRosselandOpacity( cgs_density, cgs_temperature ) * g_unitDensity * g_unitLength;

    //result = RosselandRafaOpacity (cgs_density, cgs_temperature) * g_unitDensity *g_unitLength;
    assert( result > 0. );
    return result;
}

/**
    Returns the Planck opacity in code units

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  d
    \param[in]  grid

    \returns Planck opacity in code units
*/
double ComputePlanckOpacity( int k, int j, int i, Data *d, Grid *grid ){  
     double pres_aux = d->Vc[PRS][k][j][i];
     if (pres_aux < 0.0) pres_aux = g_smallPressure;
    double conv = 11604.525;
    double cgs_density = (ComputeIntermediateValue( k, j, i, NO_SHIFT, d->Vc[RHO]) * g_unitDensity);   /**<density in log cgs units */
    double cgs_temperature = ComputeIntermediateValueWithFunction( k, j, i, NO_SHIFT, d, GetTemperature ) * g_unitTemperature; /**<temperature in log cgs units */

//    double cgs_density = ComputeIntermediateValue( k, j, i, NO_SHIFT, d->Vc[RHO] ) * g_unitDensity;                     /**<density in cgs units */
//    double cgs_temperature = ComputeIntermediateValueWithFunction( k, j, i, NO_SHIFT, d, GetTemperature ) * g_unitTemperature; /**<temperature in cgs units */
    
    double result;

    assert( cgs_density > 0. );
    assert( cgs_temperature > 0. );
    
    result = RadiationPlanckOpacity( cgs_density, cgs_temperature ) * g_unitDensity * g_unitLength;
        assert( result > 0. );
    return result;
}

                /*          MODIFICA            */

 double ComputePowerlosses( int k, int j, int i, Data *d, Grid *grid )
{   
     double pres_aux = d->Vc[PRS][k][j][i];
     if (pres_aux < 0.0) pres_aux = g_smallPressure;

    
    double conv = 11604.525;
    double cgs_density = (ComputeIntermediateValue( k, j, i, NO_SHIFT, d->Vc[RHO]) * g_unitDensity);   /**<density in log cgs units */
    double cgs_temperature = ComputeIntermediateValueWithFunction( k, j, i, NO_SHIFT, d, GetTemperature ) *g_unitTemperature; /**<temperature in log cgs units */
    double result;

    assert( cgs_density > 0. );
    assert( cgs_temperature > 0. );

    result = RadiationPowerlosses( cgs_density, cgs_temperature )/( g_unitDensity * g_unitVelocity * g_unitLength )  ;
   /* if (d->Vc[TRC][k][j][i] > ){
    result =result*(1.- (d->Vc[TRC][k][j][i]));
    }
   */
    assert( result > 0. );
    return result;
}

double ComputeDerivativePowerlosses( int k, int j, int i, Data *d, Grid *grid )
{   
     double pres_aux = d->Vc[PRS][k][j][i];
 if (pres_aux < 0.0) pres_aux = g_smallPressure;


    double conv = 11604.525;
    double cgs_density = (ComputeIntermediateValue( k, j, i, NO_SHIFT, d->Vc[RHO] )* g_unitDensity);   /**<density in log cgs units */
    double cgs_temperature = (ComputeIntermediateValueWithFunction( k, j, i,NO_SHIFT, d, GetTemperature ))*g_unitTemperature; /**<temperature in log cgs units */
    double result;
    
    assert( cgs_density > 0. );
    assert( cgs_temperature > 0. );
    
    result = RadiationDerivativePowerloss( cgs_density, cgs_temperature )/((g_unitDensity * g_unitVelocity * g_unitLength )/g_unitTemperature);
    /* if (d->Vc[TRC][k][j][i] > 0.1){
      result =result*(1.- (d->Vc[TRC][k][j][i]));
     }*/

    //printf("RESULT = %e \n", result);
    assert( result > 0. );
    return result;
}

                    /*             FINE                */

/**
    Returns the acceleration caused by the radiation.
    \f[
        \vec{a}_R = - \frac{\lambda}{\rho} \nabla E
    \f]

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  grid
    \param[in]  v
    \param[out] a

    \returns \f$ D_{i,j,k} \f$
*/
void ComputeRadiationAcceleration( int k, int j, int i, Grid *grid, double *v, double *a )
{
    double grad_E[3];
    double abs_grad_E = 0., denominator = 0, R = 0, factor = 0.;
    double cgs_gas_temperature = 0., cgs_density = 0.;
    double ***E = radiation_data.Erad;

    grad_E[IDIR] = ( ( E[k][j][i] + E[k][j][i + 1] ) / 2.0 ) - ( ( E[k][j][i - 1] + E[k][j][i] ) / 2.0 );
    grad_E[JDIR] = ( ( E[k][j][i] + E[k][j + 1][i] ) / 2.0 ) - ( ( E[k][j - 1][i] + E[k][j][i] ) / 2.0 );
    grad_E[KDIR] = ( ( E[k][j][i] + E[k + 1][j][i] ) / 2.0 ) - ( ( E[k - 1][j][i] + E[k][j][i] ) / 2.0 );

    #if (GEOMETRY == CARTESIAN)
        grad_E[IDIR] /= grid[IDIR].dx[i];
        grad_E[JDIR] /= grid[JDIR].dx[j];
        grad_E[KDIR] /= grid[KDIR].dx[k];
    #elif (GEOMETRY == POLAR)
        grad_E[IDIR] /= grid[IDIR].dx[i];
        grad_E[JDIR] /= ( grid[JDIR].dx[j] * grid[IDIR].x[i] );
        grad_E[KDIR] /= grid[KDIR].dx[k];
    #elif (GEOMETRY == CYLINDRICAL)
        #ERROR CYLINDRICAL not implemented so far
    #elif (GEOMETRY == SPHERICAL)
        grad_E[IDIR] /= grid[IDIR].dx[i];
        grad_E[JDIR] /= ( grid[JDIR].dx[j] * grid[IDIR].x[i] );
        grad_E[KDIR] /= ( grid[KDIR].dx[k] * grid[IDIR].x[i] * sin( grid[JDIR].x[j] ) );
    #endif

    abs_grad_E = sqrt( POW2( grad_E[IDIR] ) + POW2( grad_E[JDIR] ) + POW2( grad_E[KDIR] ) );

    cgs_gas_temperature = ( ( ( v[PRS] * radiation_data.mu * CONST_mp ) / ( v[RHO] * CONST_kB ) ) * POW2( g_unitVelocity ) );
    cgs_density = v[RHO] * g_unitDensity;

    #if RADIATION_SMALL_NUMBERS_FIX == YES
        R = abs_grad_E / ( ( RadiationRosselandOpacity( cgs_density, cgs_gas_temperature ) * g_unitDensity * g_unitLength ) * v[RHO] * E[k][j][i] + RADIATION_SMALL_NUMBERS_FIX_VALUE );
    #else
        R = abs_grad_E / ( ( RadiationRosselandOpacity( cgs_density, cgs_gas_temperature ) * g_unitDensity * g_unitLength ) * v[RHO] * E[k][j][i] );
    #endif

    factor = -1.0 * ( Fluxlimiter( R ) / v[RHO] );
    a[IDIR] = factor * grad_E[IDIR];
    a[JDIR] = factor * grad_E[JDIR];
    a[KDIR] = factor * grad_E[KDIR];
}

/**
    Returns the artificial dissipation. In general the artificial dissipation \f$ D_{i,j,k} \f$ is computed with the equation
    \f[
    D_{i,j,k} = r_i^2 \underbrace{\rho_{i,j,k} \nu}_{\eta_{i,j,k}} \left(\frac{\partial \Omega_{i,j,k}}{\partial r_i}\right)^2
    \f]
    where \f$ \nu \f$ is the kinematic viscosity and \f$ \Omega_{i,j,k} \f$ the angular velocity.
    There are now two options to compute \f$ D_{i,j,k} \f$. One option is to use as angular velocity the keplerian angular
    velocity
    \f[
        \Omega_{i,j,k} = \sqrt{\frac{G (m_\odot + m_p)}{r^3}}
    \f]
    with which we get:
    \f[
    D_{i,j,k} = \frac{9}{4} \rho_{i,j,k} \nu \frac{G (m_\odot + m_p)}{r_i^3}
    \f]
    \n
    Or in the other case which we are using here. We use for the angular velocity \f$ \Omega_{i,j,k} = \frac{u}{r} \f$ where \f$ u = v_\phi \f$ and
    compute the derivation numerically. By using
    \f[
    \frac{\partial u}{\partial r} \approx \frac{u(r+\Delta r) - u(r-\Delta r)}{2 \Delta r}
    \f]

    \f[
    \frac{\partial \Omega_{i,j,k}}{\partial r_i} = \frac{\partial u_{i,j,k}}{\partial r_i} r^{-1} - u_{i,j,k}(r_i) r^{-2}
    \f]
    we get
    \f[
    D_{i,j,k} = r_i^2 \rho_{i,j,k} \nu \left(\frac{u_{i+1,j,k} - u_{i-1,j,k}}{2 \Delta r} r_i^{-1} - u_{i,j,k} r_i^{-2}\right)^2
    \f]

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  shift
    \param[in]  d
    \param[in]  grid

    \returns \f$ D_{i,j,k} \f$
*/
double ComputeArtificialDissipation( int k, int j, int i, Data *data, Grid *grid )
{
    #if ARTIFICIAL_DISSIPATION == YES
        #if GEOMETRY == SPHERICAL
            double r = grid[IDIR].x[i], r1 = 1.0 / r, r2 = r1 * r1, dr = grid[IDIR].dx[i];
            double dOmega;

            dOmega = ( ( data->Vc[VZ][k][j][i + 1] - data->Vc[VZ][k][j][i - 1] ) / ( 2.0 * dr ) ) * r1 - data->Vc[VZ][k][j][i] * r2;

            //BEGIN calculation of the viscosity
            double eta1_visc = 0.0, eta2_visc = 0.0;
            int n = 0;
            double tmp[NVAR];

            for( n = 0; n < NVAR; ++n ) tmp[n] = data->Vc[n][k][j][i];
            eta_visc_func( tmp, grid[IDIR].x[i], grid[JDIR].x[j], grid[KDIR].x[k], &eta1_visc, &eta2_visc );
            //END

            return POW2( r ) * eta1_visc * POW2( dOmega );
        #endif
    #endif
    return 0.0;
}

/**
    Returns the coefficient K^n_{i,j,k} in code units

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  shift
    \param[in]  d
    \param[in]  grid

    \returns
        \f[
        K^n_{i,j,k} = \frac{c \lambda}{\kappa_R \rho}
        \f]
        where \f$ \lambda \f$ = fluxlimiter() and \f$ \kappa_R \f$ = ComputeRosselandOpacity()
*/
double ComputeCoefficientK( int k, int j, int i, int shift, Data *d, Grid *grid )
{
    return ( radiation_data.code_c * Fluxlimiter( FluxlimiterR( k, j, i, shift, d, grid ) ) ) / ( ComputeRosselandOpacity( k, j, i, shift, d, grid ) * ComputeIntermediateValue( k, j, i, shift, d->Vc[RHO] ) );
}

/**
    Computes the geometry coefficient \f$ G^r_{x_1} \f$ in code units

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  grid

    \returns
        if geometry is cartesian
        \f[
            G^r_{x_1} = \frac{1}{x_{i+\frac{1}{2}} - x_{i-\frac{1}{2}}}
        \f]
        if geometry is polar
        \f[
            G^r_{x_1} = \frac{2r_{i+\frac{1}{2} }}{r_{i+\frac{1}{2} }^2 - r_{i-\frac{1}{2} }^2}
        \f]
        if geometry is spherical
        \f[
            G^r_{x_1} = \frac{r_{i+\frac{1}{2} }^2 \left(\cos \theta_{j-\frac{1}{2}}-\cos \theta_{j+\frac{1}{2}}\right) \Delta \phi_k }{V_{i,j,k}}
        \f]
*/
double ComputeGeometryCoefficientGrx1( int k, int j, int i, Grid *grid )
{
    #if (GEOMETRY == CARTESIAN)
        return 1. / ( grid[IDIR].xr[i] - grid[IDIR].xl[i] );
    #elif (GEOMETRY == POLAR)
        return ( 2. * grid[IDIR].xr[i] ) / ( POW2( grid[IDIR].xr[i] ) - POW2( grid[IDIR].xl[i] ) );
    #elif (GEOMETRY == CYLINDRICAL)
        #ERROR CYLINDRICAL not implemented so far
    #elif (GEOMETRY == SPHERICAL)
        return ( POW2( grid[IDIR].xr[i] ) * ( cos( grid[JDIR].xl[j] ) - cos( grid[JDIR].xr[j] ) ) * grid[KDIR].dx[k] ) / ( /*V_{ijk}*/grid[IDIR].dV[i] * grid[JDIR].dV[j] * grid[KDIR].dV[k] );
    #endif
}

/**
    Computes the geometry coefficient \f$ G^l_{x_1} \f$ in code units

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  grid

    \returns
        if geometry is cartesian
        \f[
            G^l_{x_1} = G^r_{x_1} = \frac{1}{x_{i+\frac{1}{2}} - x_{i-\frac{1}{2}}}
        \f]
        if geomentry is polar
        \f[
            G^l_{x_1} = \frac{2r_{i-\frac{1}{2} }}{r_{i+\frac{1}{2} }^2 - r_{i-\frac{1}{2} }^2}
        \f]
        if geometry is spherical
        \f[
            G^r_{x_1} = \frac{r_{i-\frac{1}{2} }^2 \left(\cos \theta_{j-\frac{1}{2}}-\cos \theta_{j+\frac{1}{2}}\right)  \Delta \phi_k }{V_{i,j,k}}
        \f]
*/
double ComputeGeometryCoefficientGlx1( int k, int j, int i, Grid *grid )
{
    #if (GEOMETRY == CARTESIAN)
        return 1. / ( grid[IDIR].xr[i] - grid[IDIR].xl[i] );
    #elif (GEOMETRY == POLAR)
        return ( 2. * grid[IDIR].xl[i] ) / ( POW2( grid[IDIR].xr[i] ) - POW2( grid[IDIR].xl[i] ) );
    #elif (GEOMETRY == CYLINDRICAL)
        #ERROR CYLINDRICAL not implemented so far
    #elif (GEOMETRY == SPHERICAL)
        return ( POW2( grid[IDIR].xl[i] ) * ( cos( grid[JDIR].xl[j] ) - cos( grid[JDIR].xr[j] ) ) * grid[KDIR].dx[k] ) / ( /*V_{ijk}*/grid[IDIR].dV[i] * grid[JDIR].dV[j] * grid[KDIR].dV[k] );
    #endif
}

/**
    Computes the geometry coefficient \f$ G^r_{x_2} \f$ in code units

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  grid

    \returns
        if geometry is cartesian
        \f[
            G^r_{x_2} = \frac{1}{y_{j+\frac{1}{2}} - y_{j-\frac{1}{2}}}
        \f]
        if geometry is polar
        \f[
            G^r_{x_2} = \frac{2}{r_i^2 \Delta \phi_j}
        \f]
        the old one was
        \f[
            G^r_{x_2} = \frac{\log r_{i+\frac{1}{2} } - \log r_{i-\frac{1}{2} }}{\frac{1}{2}\left(r_{i+\frac{1}{2} }^2 - r_{i-\frac{1}{2} }^2\right) \left(\phi_{j+\frac{1}{2} } + \phi_{j-\frac{1}{2} }\right)}
        \f]
        if geometry is spherical
        \f[
            G^r_{x_2} = \frac{\Delta r_i \Delta \phi_k \sin \theta_{j + \frac{1}{2}}}{V_{i,j,k}}
        \f]
*/
double ComputeGeometryCoefficientGrx2( int k, int j, int i, Grid *grid )
{
    #if (GEOMETRY == CARTESIAN)
        return 1. / ( grid[JDIR].xr[j] - grid[JDIR].xl[j] );
    #elif (GEOMETRY == POLAR)
    //  return 2.0 / (grid[IDIR].x[i] * grid[JDIR].dx[j]);
        return ( log( grid[IDIR].xr[i] ) - log( grid[IDIR].xl[i] ) ) / ( 0.5 * ( POW2( grid[IDIR].xr[i] ) - POW2( grid[IDIR].xl[i] ) ) * ( grid[JDIR].xr[j] - grid[JDIR].xl[j] ) );
    #elif (GEOMETRY == CYLINDRICAL)
        #ERROR CYLINDRICAL not implemented so far
    #elif (GEOMETRY == SPHERICAL)
        return ( grid[IDIR].dx[i] * grid[KDIR].dx[k] * sin( grid[JDIR].xr[j] ) ) / ( /*V_{ijk}*/grid[IDIR].dV[i] * grid[JDIR].dV[j] * grid[KDIR].dV[k] );
    #endif
}

/**
    Computes the geometry coefficient \f$ G^l_{x_2} \f$ in code units

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  grid

    \returns
        if geometry is cartesian
        \f[
            G^l_{x_2} = G^r_{x_2} = \frac{1}{y_{j+\frac{1}{2}} - y_{j-\frac{1}{2}}}
        \f]
        if geometry is polar
        \f[
            G^r_{x_2} = \frac{2}{r_i^2 \Delta \phi_j}
        \f]
        the old one was
        \f[
            G^r_{x_2} = \frac{\log r_{i+\frac{1}{2} } - \log r_{i-\frac{1}{2} }}{\frac{1}{2}\left(r_{i+\frac{1}{2} }^2 - r_{i-\frac{1}{2} }^2\right) \left(\phi_{j+\frac{1}{2} } + \phi_{j-\frac{1}{2} }\right)}
        \f]
        if geometry is spherical
        \f[
            G^r_{x_2} = \frac{\Delta r_i \Delta \phi_k \sin \theta_{j - \frac{1}{2}}}{V_{i,j,k}}
        \f]
*/
double ComputeGeometryCoefficientGlx2( int k, int j, int i, Grid *grid )
{
    #if (GEOMETRY == CARTESIAN)
        return 1. / ( grid[JDIR].xr[j] - grid[JDIR].xl[j] );
    #elif (GEOMETRY == POLAR)
    //  return 2.0 / (grid[IDIR].x[i] * grid[JDIR].dx[j]);
        return ( log( grid[IDIR].xr[i] ) - log( grid[IDIR].xl[i] ) ) / ( 0.5 * ( POW2( grid[IDIR].xr[i] ) - POW2( grid[IDIR].xl[i] ) ) * ( grid[JDIR].xr[j] - grid[JDIR].xl[j] ) );
    #elif (GEOMETRY == CYLINDRICAL)
        #ERROR CYLINDRICAL not implemented so far
    #elif (GEOMETRY == SPHERICAL)
        return ( grid[IDIR].dx[i] * grid[KDIR].dx[k] * sin( grid[JDIR].xl[j] ) ) / ( /*V_{ijk}*/grid[IDIR].dV[i] * grid[JDIR].dV[j] * grid[KDIR].dV[k] );
    #endif
}

/**
    Computes the geometry coefficient \f$ G^r_{x_3} \f$ in code units

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  grid

    \returns
        if geometry is cartesian
        \f[
            G^r_{x_3} = \frac{1}{z_{k+\frac{1}{2}} - z_{k-\frac{1}{2}}}
        \f]
        if geometry is polar
        \f[
            G^r_{x_3} = \frac{1}{z_{k+\frac{1}{2} } - z_{k-\frac{1}{2} }}
        \f]
        if geometry is spherical
        \f[
            G^r_{x_3} = \frac{\Delta r_i}{V_{i,j,k}} \frac{\theta_{j+\frac{1}{2}}-\theta_{j-\frac{1}{2}}}{\sin \theta_j}
        \f]
        the old one was
        \f[
            G^r_{x_3} = \frac{\Delta r_i}{V_{i,j,k}} \left(\log\left(\tan\frac{\theta_{j+\frac{1}{2} }}{2}\right)-\log\left(\tan\frac{\theta_{j-\frac{1}{2} }}{2}\right)\right)
        \f]
*/
double ComputeGeometryCoefficientGrx3( int k, int j, int i, Grid *grid )
{
    #if (GEOMETRY == CARTESIAN)
        return 1. / ( grid[KDIR].xr[k] - grid[KDIR].xl[k] );
    #elif (GEOMETRY == POLAR)
        return 1. / ( grid[KDIR].xr[k] - grid[KDIR].xl[k] );
    #elif (GEOMETRY == CYLINDRICAL)
        #ERROR CYLINDRICAL not implemented so far
    #elif (GEOMETRY == SPHERICAL)
        return ( grid[IDIR].dx[i] * ( grid[JDIR].xr[j] - grid[JDIR].xl[j] ) ) / ( /*V_{ijk}*/grid[IDIR].dV[i] * grid[JDIR].dV[j] * grid[KDIR].dV[k] );
//         return (grid[IDIR].dx[i] * (log(tan(0.5*grid[JDIR].xr[j])) - log(tan(0.5*grid[JDIR].xl[j])))) / (/*V_{ijk}*/grid[IDIR].dV[i] * grid[JDIR].dV[j] * grid[KDIR].dV[k]);
    #endif
}

/**
    Computes the geometry coefficient \f$ G^l_{x_3} \f$ in code units

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  grid

    \returns
        if geometry is cartesian
        \f[
            G^l_{x_3} = G^r_{x_3} = \frac{1}{z_{k+\frac{1}{2}} - z_{k-\frac{1}{2}}}
        \f]
        if geometry is polar
        \f[
            G^l_{x_3} = G^r_{x_3} = \frac{1}{z_{k+\frac{1}{2} } - z_{k-\frac{1}{2} }}
        \f]
        if geometry is spherical
        \f[
            G^r_{x_3} = \frac{\Delta r_i}{V_{i,j,k}} \frac{\theta_{j+\frac{1}{2}}-\theta_{j-\frac{1}{2}}}{\sin \theta_j}
        \f]
        the old one was
        \f[
            G^r_{x_3} = \frac{\Delta r_i}{V_{i,j,k}} \left(\log\left(\tan\frac{\theta_{j+\frac{1}{2} }}{2}\right)-\log\left(\tan\frac{\theta_{j-\frac{1}{2} }}{2}\right)\right)
        \f]
*/
double ComputeGeometryCoefficientGlx3( int k, int j, int i, Grid *grid )
{
    #if (GEOMETRY == CARTESIAN)
        return 1. / ( grid[KDIR].xr[k] - grid[KDIR].xl[k] );
    #elif (GEOMETRY == POLAR)
        return 1. / ( grid[KDIR].xr[k] - grid[KDIR].xl[k] );
    #elif (GEOMETRY == CYLINDRICAL)
        #ERROR CYLINDRICAL not implemented so far
    #elif (GEOMETRY == SPHERICAL)
        return ( grid[IDIR].dx[i] * ( grid[JDIR].xr[j] - grid[JDIR].xl[j] ) ) / ( /*V_{ijk}*/grid[IDIR].dV[i] * grid[JDIR].dV[j] * grid[KDIR].dV[k] );
    //  return (grid[IDIR].dx[i] * (log(tan(0.5*grid[JDIR].xr[j])) - log(tan(0.5*grid[JDIR].xl[j])))) / (/*V_{ijk}*/grid[IDIR].dV[i] * grid[JDIR].dV[j] * grid[KDIR].dV[k]);
    #endif
}

/**
    Computes the coefficient \f$ U^1_{i,j,k} \f$ in code units

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  d
    \param[in]  grid

    \returns
        \f[
            U^1_{i,j,k} = 1 -U^7_{i,j,k} - U^6_{i,j,k} - U^5_{i,j,k} - U^4_{i,j,k} - U^3_{i,j,k} - U^2_{i,j,k} + \frac{\rho_{i,j,k}^n {\kappa_P^n}_{i,j,k} c \Delta t c_V}{c_V + 4 a_R (T_{i,j,k}^n)^3 {\kappa_P^n}_{i,j,k} c \Delta t}
        \f]

*/
double ComputeCoefficientU1( int k, int j, int i, Data *d, Grid *grid )
{
    double plank_opacity = ComputePlanckOpacity( k, j, i, d, grid );
    double temperature = GetTemperature( k, j, i, d );
    double c_V = ComputeSpecificHeatCapacity( k, j, i, d );
    double delta_t = g_dt/(g_unitLength/g_unitVelocity);
    double d_pwl = ComputeDerivativePowerlosses( k, j, i, d, grid);
    
    double coef1 = plank_opacity * d->Vc[RHO][k][j][i] * radiation_data.code_c * delta_t;
    double coef2 = delta_t/(d->Vc[RHO][k][j][i]*c_V) * d_pwl;
    // NLTE
    double summand;
    
    summand = coef1 / ( 1. + coef2 );

    
    return 1. - ComputeCoefficientU7( k, j, i, d, grid ) - ComputeCoefficientU6( k, j, i, d, grid ) - ComputeCoefficientU5( k, j, i, d, grid ) - ComputeCoefficientU4( k, j, i, d, grid ) - ComputeCoefficientU3( k, j, i, d, grid ) - ComputeCoefficientU2( k, j, i, d, grid ) + summand;
     
}

/**
    \copydoc ComputeCoefficientU1
*/
double ComputeCoefficientU1Advanced( int k, int j, int i, Data *d, Grid *grid, double U2, double U3, double U4, double U5, double U6, double U7 )
{
    double plank_opacity = ComputePlanckOpacity( k, j, i, d, grid );
    double temperature = GetTemperature( k, j, i, d );
    double c_V = ComputeSpecificHeatCapacity( k, j, i, d );
    double delta_t = g_dt/(g_unitLength/g_unitVelocity);
    double d_pwl = ComputeDerivativePowerlosses( k, j, i, d, grid);

    double coef1 = plank_opacity * d->Vc[RHO][k][j][i] * radiation_data.code_c * delta_t;
    double coef2 = delta_t/(d->Vc[RHO][k][j][i]*c_V) * d_pwl;

  
// NLTE 
    double summand;
    
       summand = coef1 / ( 1. + coef2 );
    
// LTE
   // double tmp = plank_opacity * radiation_data.code_c * g_dt;
   // double summand = ( d->Vc[RHO][k][j][i] * tmp * c_V ) / ( c_V + 4.0 * radiation_data.code_aR * POW3( temperature ) * tmp );
 
    return 1. - U7 - U6 - U5 - U4 - U3 - U2 + summand;
}

/**
    Computes the coefficient \f$ U^2_{i,j,k} \f$ in code units

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  d
    \param[in]  grid

    \returns
        \f[
            U^2_{i,j,k} = - \frac{G^l_{x_3} K_{i,j,k-\frac{1}{2}}^n \Delta t}{\Delta {x_3}_{k-\frac{1}{2}}}
        \f]
*/
double ComputeCoefficientU2( int k, int j, int i, Data *d, Grid *grid )
{
    double dx = fabs( grid[KDIR].x[k] - grid[KDIR].x[k - 1] );
    return ( -1. * ComputeGeometryCoefficientGlx3( k, j, i, grid ) * ComputeCoefficientK( k, j, i, KDIR_mOH, d, grid ) * g_dt ) / ( dx );
}

/**
    Computes the coefficient \f$ U^3_{i,j,k} \f$ in code units

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  d
    \param[in]  grid

    \returns
        \f[
            U^3_{i,j,k} = - \frac{G^r_{x_3} K_{i,j,k+\frac{1}{2}}^n \Delta t}{\Delta {x_3}_{k+\frac{1}{2}}}
        \f]
*/
double ComputeCoefficientU3( int k, int j, int i, Data *d, Grid *grid )
{
    double dx = fabs( grid[KDIR].x[k + 1] - grid[KDIR].x[k] );
    return ( -1. * ComputeGeometryCoefficientGrx3( k, j, i, grid ) * ComputeCoefficientK( k, j, i, KDIR_pOH, d, grid ) * g_dt ) / ( dx );
}

/**
    Computes the coefficient \f$ U^4_{i,j,k} \f$ in code units

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  d
    \param[in]  grid

    \returns
        \f[
            U^4_{i,j,k} = - \frac{G^l_{x_2} K_{i,j-\frac{1}{2},k}^n \Delta t}{\Delta {x_2}_{j-\frac{1}{2}}}
        \f]
*/
double ComputeCoefficientU4( int k, int j, int i, Data *d, Grid *grid )
{
    double dx = fabs( grid[JDIR].x[j] - grid[JDIR].x[j - 1] );
    return ( -1. * ComputeGeometryCoefficientGlx2( k, j, i, grid ) * ComputeCoefficientK( k, j, i, JDIR_mOH, d, grid ) * g_dt ) / ( dx );
}

/**
    Computes the coefficient \f$ U^5_{i,j,k} \f$ in code units

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  d
    \param[in]  grid

    \returns
        \f[
            U^5_{i,j,k} = - \frac{G^r_{x_2} K_{i,j+\frac{1}{2},k}^n \Delta t}{\Delta {x_2}_{j+\frac{1}{2}}}
        \f]
*/
double ComputeCoefficientU5( int k, int j, int i, Data *d, Grid *grid )
{
    double dx = fabs( grid[JDIR].x[j + 1] - grid[JDIR].x[j] );
    return ( -1. * ComputeGeometryCoefficientGrx2( k, j, i, grid ) * ComputeCoefficientK( k, j, i, JDIR_pOH, d, grid ) * g_dt ) / ( dx );
}

/**
    Computes the coefficient \f$ U^6_{i,j,k} \f$ in code units

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  d
    \param[in]  grid

    \returns
        \f[
            U^6_{i,j,k} = - \frac{G^l_{x_1} K_{i-\frac{1}{2},j,k}^n \Delta t}{\Delta {x_1}_{i-\frac{1}{2}}}
        \f]
*/
double ComputeCoefficientU6( int k, int j, int i, Data *d, Grid *grid )
{
    double dx = fabs( grid[IDIR].x[i] - grid[IDIR].x[i - 1] );
    return ( -1. * ComputeGeometryCoefficientGlx1( k, j, i, grid ) * ComputeCoefficientK( k, j, i, IDIR_mOH, d, grid ) * g_dt ) / ( dx );
}

/**
    Computes the coefficient \f$ U^7_{i,j,k} \f$ in code units

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  d
    \param[in]  grid

    \returns
        \f[
            U^7_{i,j,k} = - \frac{G^r_{x_1} K_{i+\frac{1}{2},j,k}^n \Delta t}{\Delta {x_1}_{i+\frac{1}{2}}}
        \f]
*/
double ComputeCoefficientU7( int k, int j, int i, Data *d, Grid *grid )
{
    double dx = fabs( grid[IDIR].x[i + 1] - grid[IDIR].x[i] );
    return ( -1. * ComputeGeometryCoefficientGrx1( k, j, i, grid ) * ComputeCoefficientK( k, j, i, IDIR_pOH, d, grid ) * g_dt ) / ( dx );
}

/**
    Returns the right hand side of the equation \f$ B_{i,j,k} \f$ in code units.

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  d
    \param[in]  grid

    \returns
        In the case that irradiation and/or artificial dissipation is used this function returnes
        \f[
            B_{i,j,k} = \frac{{\kappa_P}_{i,j,k}^n c a_R (T_{i,j,k}^n)^3 \Delta t}{c_V + {4\kappa_P}_{i,j,k}^n c a_R (T_{i,j,k}^n)^3 \Delta t} \left(4 W_{i,j,k}^n \Delta t +\rho_{i,j,k}^n T_{i,j,k}^n c_V\right)+ E_{i,j,k}^n
        \f]
        where \f$ W^n_{i,j,k} \f$ is either \f$ S^n_{i,j,k} \f$ in the case of irradiation or \f$ D^n_{i,j,k} \f$ in the case of artificial dissipation. If irradiation and artificial dissipation is used together then \f$ W^n_{i,j,k} \f$ is the summ of both \f$ W^n_{i,j,k} = S^n_{i,j,k}+D^n_{i,j,k}\f$.\n
        else
        \f[
            B_{i,j,k} = \frac{\rho_{i,j,k}^n {\kappa_P}_{i,j,k}^n c a_R (T_{i,j,k}^n)^4 \Delta t c_V}{c_V + 4 {\kappa_P}_{i,j,k}^n c a_R (T_{i,j,k}^n)^3 \Delta t} + E_{i,j,k}^n
        \f]
*/
double ComputeCoefficientB( int k, int j, int i, Data *d, Grid *grid )
{
    #if IRRADIATION == YES || ARTIFICIAL_DISSIPATION == YES
        double plank_opacity = ComputePlanckOpacity( k, j, i, d, grid );
        double temperature = GetTemperature( k, j, i, d );
        double rho = d->Vc[RHO][k][j][i];
        double c_V = ComputeSpecificHeatCapacity( k, j, i, d );
        double tmp = plank_opacity * radiation_data.code_c * radiation_data.code_aR * POW3( temperature ) * g_dt;
        double W = 0.0;

        #if IRRADIATION == YES
            W = irradiation.S[k][j][i];
        #endif

        #if ARTIFICIAL_DISSIPATION == YES
            W += ComputeArtificialDissipation( k, j, i, d, grid );
        #endif

        return ( tmp / ( c_V + 4.0 * tmp ) ) * ( 4.0 * W * g_dt + rho * temperature * c_V ) + radiation_data.Erad[k][j][i];
    #else
        double rho = d->Vc[RHO][k][j][i];
        double plank_opacity = ComputePlanckOpacity( k, j, i, d, grid );
        double temperature = GetTemperature( k, j, i, d );
        double c_V = ComputeSpecificHeatCapacity( k, j, i, d );
        double pwl = ComputePowerlosses( k, j, i, d, grid );
        double delta_t = g_dt/(g_unitLength/g_unitVelocity);
        double d_pwl = ComputeDerivativePowerlosses( k, j, i, d, grid )*3.1e-10/(rho*ComputePlanckOpacity( k, j, i, d, grid ));
        double coef2 = ( delta_t/(d->Vc[RHO][k][j][i]*c_V) ) * d_pwl;
	    double B;
       //   NLTE
    
        B =   ( (pwl* delta_t)/(1. + coef2) ) + radiation_data.Erad[k][j][i];
    
        return B;


    
    
    #endif
}
/**
    Calculates the actual temperature \f$ T_{i,j,k}^n \f$ in code units

    \f[
        T_{i,j,k}^n = \frac{p}{\rho} \frac{\mu m_u}{k_B}
    \f]


    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  d

    \returns    \f$ T_{i,j,k}^n \f$

*/
double GetTemperature( int k, int j, int i, Data *d )
{
    double temperature_cgs = ( ( d->Vc[PRS][k][j][i] * radiation_data.mu * CONST_mp ) / ( d->Vc[RHO][k][j][i] * CONST_kB ) ) * POW2( g_unitVelocity ); //temperature in Kelvin
    return temperature_cgs / g_unitTemperature;
}

/**
    Returns the specific heat capacity \f$ c_V \f$ in code units

    \note
        The heat capacity is given by \f$ C_V=\rho c_V \f$

    \param[in]  k
    \param[in]  j
    \param[in]  i
    \param[in]  d

    \returns
        \f[
            c_V = \frac{k_B}{(\gamma - 1) \mu m_H}
        \f]
 */
double ComputeSpecificHeatCapacity( int k, int j, int i, Data *d )
{
    return ( radiation_data.code_kB ) / ( ( g_gamma - 1. ) * radiation_data.mu * radiation_data.code_mH );
}
